#include "pch.h"

#include "AlAsSb.h"
//------------------------------------------------------------------------------
double AlAsSb::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.a(iT) + mSb * mAlSb.a(iT) );
}
//------------------------------------------------------------------------------
double AlAsSb::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	if (iPoint == 'G') return ( mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.8 );
	else if (iPoint == 'X') return ( mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.28 );
	else if (iPoint == 'L') return ( mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.28 );
	else return 0.;
}
//------------------------------------------------------------------------------
double AlAsSb::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double AlAsSb::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.Dso() + mSb * mAlSb.Dso() - mAs * mSb * 0.15 );
}
//------------------------------------------------------------------------------
double AlAsSb::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	}
	return ( mAs * mAlAs.me() + mSb * mAlSb.me() );
}
//------------------------------------------------------------------------------
double AlAsSb::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	}
	return ( mAs * mAlAs.mhh() + mSb * mAlSb.mhh() );
}
//------------------------------------------------------------------------------
double AlAsSb::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	}
	return ( mAs * mAlAs.mlh() + mSb * mAlSb.mlh() );
}
//------------------------------------------------------------------------------
double AlAsSb::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.mso() + mSb * mAlSb.mso() );
}
//------------------------------------------------------------------------------
double AlAsSb::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double AlAsSb::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb; punktem odniesienia jest Ev=0eV dla InSb";
	}
	return ( mAs * mAlAs.Ev() + mSb * mAlSb.Ev() - mAs * mSb * (-1.71) );
}
//------------------------------------------------------------------------------
double AlAsSb::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double AlAsSb::ac(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.ac() + mSb * mAlSb.ac() );
}
//------------------------------------------------------------------------------
double AlAsSb::av(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.av() + mSb * mAlSb.av() );
}
//------------------------------------------------------------------------------
double AlAsSb::b(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.b() + mSb * mAlSb.b() );
}
//------------------------------------------------------------------------------
double AlAsSb::d(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.d() + mSb * mAlSb.d() );
}
//------------------------------------------------------------------------------
double AlAsSb::c11(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.c11() + mSb * mAlSb.c11() );
}
//------------------------------------------------------------------------------
double AlAsSb::c12(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.c12() + mSb * mAlSb.c12() );
}
//------------------------------------------------------------------------------
double AlAsSb::c44(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb";
	}
	return ( mAs * mAlAs.c44() + mSb * mAlSb.c44() );
}
//------------------------------------------------------------------------------
double AlAsSb::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double AlAsSb::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double AlAsSb::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double AlAsSb::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	} 
	return ( 1./(mAs/mAlAs.k(iT) + mSb/mAlSb.k(iT) + mAs*mSb*0.91) );
}
//------------------------------------------------------------------------------
double AlAsSb::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	}
	return ( mAs * mAlAs.cp(iT) + mSb * mAlSb.cp(iT) );
}
//------------------------------------------------------------------------------
double AlAsSb::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i AlSb"; 
	}
	return ( mAs * mAlAs.g(iT) + mSb * mAlSb.g(iT) );
}
//------------------------------------------------------------------------------
double AlAsSb::nR(double iLam, double iT, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="C. Alibert, M. Skouri, A. Joullie, M. Benouna, and S. Sadiq";
		ipRef->title="Refractive indices of AlSb and GaSb lattice matched AlxGa1-xAsySb1-y in the transparent wavelength region";
		ipRef->journal="Journal of Applied Physics, vol. 69, no. 5, pp. 3208-3211, 1991";
		ipRef->comment="zaleznosc poprawna dla przedzialu 0.5-7 um, autor zaleznosci: Lukasz Piskorski";
	}
    double tLam2 = iLam*iLam*1e-6;
    double tnR296K = sqrt(1.+(7.055+1.8424*mSb)*tLam2/(tLam2-(0.068+0.089*mSb)));
    return ( tnR296K + tnR296K*(mAs*4.6e-5+mSb*8.2e-5)*(iT-296.) ); // 4.6e-5 -> see: AlAs, 8.2e-5 -> Adachi (2005) ebook p.243 tab. 10.6
}
//------------------------------------------------------------------------------
double AlAsSb::abs(double iLam, double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="zalozono, ze abs. bedzie taka sama jak dla GaSb, stosowac dla AlAsSb dop. siec. do GaSb";
	}
	if (iDop == "Te")
	{
		double tN = iN*1e-24;
		double tLam = iLam*1e-3;
		double tFCabs = 2.42*tN*pow(tLam,2.16-0.22*tN); // (dopasowanie do danych: Lukasz)
		double tIVCBabs = (24.1*tN+12.5)*(1.24/tLam-(0.094*tN+0.12))+(-2.05*tN-0.37); // (dopasowanie do danych: Lukasz)
		if (tIVCBabs>0) return ( tFCabs + tIVCBabs );
		else return ( tFCabs );
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
AlAsSb::AlAsSb(double iAs)
{
	mAs = iAs;
	mSb = 1.-iAs;
}
//------------------------------------------------------------------------------
AlAsSb::AlAsSb()
{  
}
//------------------------------------------------------------------------------
AlAsSb::~AlAsSb()
{
}
//------------------------------------------------------------------------------