#include "pch.h"

#include "AlGaAs.h"
//------------------------------------------------------------------------------
double AlGaAs::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.a(iT) + mGa * mGaAs.a(iT) );
}
//------------------------------------------------------------------------------
double AlGaAs::ea(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. Levinshtein, S. Rumyanstev, and M. Shur";
		ipRef->title="Handbook Series on Semiconductor Parameters";
		ipRef->journal="Vol. 2, World Scientific, New Jersey, 1999";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	if (mAl < 0.45) return ( 4.07-1.1*mAl );
	else return ( 3.64-0.14*mAl ); 
}
//------------------------------------------------------------------------------
double AlGaAs::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	if (iPoint == 'G') return ( mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mAl * mGa * (-0.127+1.310*mAl) );
	else if (iPoint == 'X') return ( mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mAl * mGa * 0.055 );
	else if (iPoint == 'L') return ( mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) );
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaAs::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double AlGaAs::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.Dso() + mGa * mGaAs.Dso() );
}
//------------------------------------------------------------------------------
double AlGaAs::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	}
	return ( mAl * mAlAs.me() + mGa * mGaAs.me() );
}
//------------------------------------------------------------------------------
double AlGaAs::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	}
	return ( mAl * mAlAs.mhh() + mGa * mGaAs.mhh() );
}
//------------------------------------------------------------------------------
double AlGaAs::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	}
	return ( mAl * mAlAs.mlh() + mGa * mGaAs.mlh() );
}
//------------------------------------------------------------------------------
double AlGaAs::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.mso() + mGa * mGaAs.mso() );
}
//------------------------------------------------------------------------------
double AlGaAs::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double AlGaAs::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs; punktem odniesienia jest Ev=0eV dla InSb";
	}
	return ( mAl * mAlAs.Ev() + mGa * mGaAs.Ev() );
}
//------------------------------------------------------------------------------
double AlGaAs::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double AlGaAs::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.ac() + mGa * mGaAs.ac() );
}
//------------------------------------------------------------------------------
double AlGaAs::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.av() + mGa * mGaAs.av() );
}
//------------------------------------------------------------------------------
double AlGaAs::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.b() + mGa * mGaAs.b() );
}
//------------------------------------------------------------------------------
double AlGaAs::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.d() + mGa * mGaAs.d() );
}
//------------------------------------------------------------------------------
double AlGaAs::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.c11() + mGa * mGaAs.c11() );
}
//------------------------------------------------------------------------------
double AlGaAs::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.c12() + mGa * mGaAs.c12() );
}
//------------------------------------------------------------------------------
double AlGaAs::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.c44() + mGa * mGaAs.c44() );
}
//------------------------------------------------------------------------------
double AlGaAs::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double AlGaAs::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double AlGaAs::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double AlGaAs::Eact(std::string iDop, reference *ipRef)
{
	if (iDop == "Si")
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="N. Chand, T. Henderson, J. Klem, W. T. Masselink, R. Fisger, Y.-C, Chang, and H. Morkoc";
			ipRef->title="Comprehensive analysis of Si-doped Al(x)Ga(1-x)As (x=0 to 1): Theory and experiments";
			ipRef->journal="Physical Review B, vol. 30, no. 8, pp. 4481-4492, 1984";
			ipRef->comment="na podstawie danych z Tab. I";
		}
		double iS = 160.818;
		double iA = 26.2146;
		double iB = 0.339053;
		double iX = mAl - 0.2;
		return ( (iX > 0.) ? (1e-3 * (pow((iS*iX),(iA*iB*pow(0.28,iB))) * exp(-iA*pow(iX,iB)) + 10.)) : 0.01 ); // fitowanie przez MW
	}
	else if (iDop == "C")
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="R. Heilman and G. Oelgart";
			ipRef->title="Ionization energy of the carbon acceptor in Al(x)Ga(1-x)As";
			ipRef->journal="Semiconductor Science and Technology, vol. 5, no. 10, pp.1040-1045, 1990";
			ipRef->comment="wzor z abstractu";
		}
		return ( 1e-3 * (26.7 + 5.56*mAl + 110.*pow(mAl,3.4)) );
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double AlGaAs::Eact(double iN, std::string iDop, reference *ipRef)
{
	if (iDop == "Si")
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="-";
			ipRef->title="-";
			ipRef->journal="-";
			ipRef->comment="energia wyznaczona tak by dla 300K uzyskac zmierzone koncentracje";
		}
		/*double tA = cFunc::Nc(300.,me());
		double tB = -1./(cPhys::kB_eV*300.);
		double tgD = 2.;
		double tC = 2*tgD*n(300., iN, iDop);
		double tD = 4.*tgD*iN;
		return ( log( tC / (tA*(tD/tC-2.)) ) / tB );*/
		double tgD = 2.;
		double tN = tgD * pow(n(300.,iN,"Si"),2. ) / ( iN - n(300.,iN,"Si") );
		return ( -cPhys::kB_eV*300.*log(tN/cFunc::Nc(300.,me())) );
	}
	else if (iDop == "C")
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="-";
			ipRef->title="-";
			ipRef->journal="-";
			ipRef->comment="energia wyznaczona tak by dla 300K uzyskac zmierzone koncentracje";
		}
		/*double tA = cFunc::Nv(300.,mh());
		double tB = -1./(cPhys::kB_eV*300.);
		double tgA = 4.;
		double tC = 2*tgA*n(300., iN, iDop);
		double tD = 4.*tgA*iN;
		return ( log( tC / (tA*(tD/tC-2.)) ) / tB );*/
		double tgA = 4.;
		double tN = tgA * pow(n(300.,iN,"C"),2. ) / ( iN - n(300.,iN,"C") );
		return ( -cPhys::kB_eV*300.*log(tN/cFunc::Nc(300.,1.5*mh())) ); // tu zmienialem - dodalem 1.5 - P 11.12.2014
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double AlGaAs::n(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if (iDop == "Si")
	{
		double tGaAs_n = iN; // (dopasowanie do danych eksp.: Lukasz)
		double t_fxA = (1.-7.8*mAl*mAl); // x < 0.35 (dopasowanie do danych eksp.: Lukasz)
		double t_fxB = (1.14*mAl-0.36); // else (dopasowanie do danych eksp.: Lukasz)
		if (mAl < 0.35) return ( tGaAs_n*t_fxA );
		else return ( tGaAs_n*t_fxB );
	}
	else if (iDop == "Si_")
		return ( cFunc::Nact('n', Eact("Si"), iN, me(), iT) );
	else if (iDop == "Si__")
		return ( cFunc::Nact('n', Eact(iN,"Si"), iN, me(), iT) );
	else if (iDop == "C")
	{
		double tGaAs_p = 0.92*iN; // (dopasowanie do danych eksp.: Lukasz)
		return ( tGaAs_p ); // t_fx=1 (dopasowanie do danych eksp.: Lukasz)
	}
	else if (iDop == "C_")
		return ( cFunc::Nact('p', Eact("C"), iN, 1.5*mh(), iT) );// tu zmienialem - dodalem 1.5 - P 11.12.2014
	else if (iDop == "C__")
		return ( cFunc::Nact('p', Eact(iN,"C"), iN, 1.5*mh(), iT) );// tu zmienialem - dodalem 1.5 - P 11.12.2014
	else if (iDop == "Be")
	{
		double tGaAs_p = iN; // (dopasowanie do danych eksp.: Lukasz)
		return ( tGaAs_p ); // t_fx=1 (zalozenie: Lukasz)
	}	
	else		
		return iN; // TODO
}
//------------------------------------------------------------------------------
double AlGaAs::mob(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if ((iDop == "Si") || (iDop == "Si_") || (iDop == "Si__"))
	{
		double tGaAs_miRT = 6600e-4/(1+pow(n(300., iN, iDop)/5e23,0.53)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		double fxA = exp(-16.*mAl*mAl); // x < 0.5 (dopasowanie do danych eksp.: Lukasz)
		double fxB = 0.054*mAl-0.009; // else (dopasowanie do danych eksp.: Lukasz)
		if (mAl < 0.5) return ( tGaAs_miRT * pow(300./iT,1.4) * fxA ); // (1.4 -> ref: rozprawa Lukasza)
		else return ( tGaAs_miRT * pow(300./iT,1.4) * fxB ); // (1.4 -> ref: rozprawa Lukasza)
	}
	else if ((iDop == "C") || (iDop == "C_") || (iDop == "C__"))
	{
		double tGaAs_miRT = 530e-4/(1+pow(n(300., iN, iDop)/1e23,0.30)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		double fx = ( (1.00 - 0.34) / (1. + pow(mAl/0.21, 3.0)) ) + 0.34; // (dopasowanie do danych eksp.: Lukasz)
		return ( tGaAs_miRT * pow(300./iT,1.25) * fx ); // (1.25 -> ref: rozprawa Lukasza)
	}
	else if (iDop == "Zn")
	{
		double tGaAs_miRT = 480e-4/(1+pow(n(300., iN, iDop)/4e23,0.47)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		double fx = 1.; // TODO
		return ( tGaAs_miRT * pow(300./iT,1.25) * fx ); // (1.25 -> ref: rozprawa Lukasza)
	}
	else if (iDop == "Be")
	{
		double tGaAs_miRT = 840e-4/(1+pow(n(300., iN, iDop)/1e22,0.28)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		double fx = 1.; // TODO
		return ( tGaAs_miRT * pow(300./iT,1.25) * fx ); // (1.25 -> ref: rozprawa Lukasza)
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double AlGaAs::ec(double iT, double iN, std::string iDop, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cPhys::e * mob(iT, iN, iDop) * n(iT, iN, iDop) );       
}
//------------------------------------------------------------------------------
double AlGaAs::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	} 
	return ( 1./(mAl/mAlAs.k(iT) + mGa/mGaAs.k(iT) + mAl*mGa*0.32) );
}
//------------------------------------------------------------------------------
double AlGaAs::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	} 
	return ( mAl * mAlAs.cp(iT) + mGa * mGaAs.cp(iT) );
}
//------------------------------------------------------------------------------
double AlGaAs::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs"; 
	} 
	return ( mAl * mAlAs.g(iT) + mGa * mGaAs.g(iT) );
}
//------------------------------------------------------------------------------
double AlGaAs::dc(reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. Levinshtein, S. Rumyanstev, and M. Shur";
		ipRef->title="Handbook Series on Semiconductor Parameters";
		ipRef->journal="Vol. 2, World Scientific, New Jersey, 1999";
		ipRef->comment="wartosc interpolowana z danych dla AlAs i GaAs";
	}
	return ( mAl * mAlAs.dc() + mGa * mGaAs.dc() );
}
//------------------------------------------------------------------------------
double AlGaAs::nR(double iLam, double iT, reference *ipRef) 
{
	/*
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac; zal. popr. do 3 um";
	}
    double tLam2 = iLam*iLam*1e-6;
    double tnR296K = sqrt(1.+(9.659-2.604*mAl)*tLam2/(tLam2-(0.137-0.069*mAl)));
    return ( tnR296K + tnR296K*(mAl*4.6e-5+mGa*4.5e-5)*(iT-296.) );
	*/

	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Leszek Frasunkiewicz";
		ipRef->title="The refractive index of AlxGa1-xAs below the band gap";
		ipRef->journal="2000 JAP Gehrsitz i 2003 JAP Skauli";
		ipRef->comment="na podstawie danych eksperymentalnych i modeli; zal. popr. do 3 um";
	}

	double x = mAl;
	double A = -0.4490233379*x + 3.25759049;
	double B = 29.22871618*pow(x, (-2.35349122*pow(x, 8.844978824)));
	double C = -304.7269552*x*x*x + 335.1918592*x*x + 194.6726344*x - 559.6098207;

	double Ad = 0.0003442176581*x*x*x - 0.0005412098145*x*x + 0.00008671640556*x + 0.0002093262406;
	double Bd = 132.1382231*exp(-8.32822628*x*x*x + 14.65496754*x*x - 7.135900438*x);
	double Cd = 117.24*x -689.06;

	double dndT = Ad*exp(Bd / (iLam + Cd));
	return ( A*exp(B / (iLam + C)) + dndT*(iT-296) );	//zwraca nR
}
//------------------------------------------------------------------------------
double AlGaAs::abs(double iLam, double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac";
	}		
    double tEgRef300 = cFunc::Varshni(1.519, 0.5405e-3, 204., 300.); // GaAs 300 K
    double tEgT = Eg(iT);
    double tDLam = 1240.*(tEgRef300-tEgT)/(tEgRef300*tEgT);
    double tLam = (iLam-tDLam)*1e-3; // teraz w um
    double tAbsp(0.);
	
	char tType = '-';
	if ( (iDop == "C") || (iDop == "Zn") || (iDop == "Be") )
		tType = 'p';
	else if ( (iDop == "Si") || (iDop == "Te") || (iDop == "Se") )
		tType = 'n';
	
	double tN = n(iT, iN, iDop);
	
	if (tType == 'n')
	{
		if (tLam <= 6.) // 0.85-6 um
			tAbsp = (tN*1e-24)*(1e24*exp(-tLam/0.0169)+4.67+0.00211*pow(tLam,4.80));
		else if (tLam <= 27.) // 6-27 um
			tAbsp = (tN*1e-24)*(-8.4+0.233*pow(tLam,2.6));
	}
	else if (tType == 'p')
	{
		if (tLam <= 6.) // 0.85-6 um
			tAbsp = (tN*1e-24)*(1e24*exp(-tLam/0.0173)+0.114*pow(tLam,4.00)+73.*exp(-0.76*pow(tLam-2.74,2.)));
		else if (tLam <= 27.) // 6-27 um
			tAbsp = (tN*1e-24)*(0.589*pow(tLam,3.)-22.87*pow(tLam,2.)+308.*tLam-1004.14);
	}

    return ( tAbsp );
}
//------------------------------------------------------------------------------
AlGaAs::AlGaAs(double iAl)
{
	mAl = iAl;
	mGa = 1.-iAl;
}
//------------------------------------------------------------------------------
AlGaAs::AlGaAs()
{  
}
//------------------------------------------------------------------------------
AlGaAs::~AlGaAs()
{
}
//------------------------------------------------------------------------------