#include "pch.h"

#include "AlGaAsSb.h"
//------------------------------------------------------------------------------
double AlGaAsSb::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	return ( mAl*mAs*mAlAs.a(iT) + mAl*mSb*mAlSb.a(iT) + mGa*mAs*mGaAs.a(iT) + mGa*mSb*mGaSb.a(iT) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tEgLin = mAl*mAs*mAlAs.Eg(iT, iPoint) + mAl*mSb*mAlSb.Eg(iT, iPoint) + mGa*mAs*mGaAs.Eg(iT, iPoint) + mGa*mSb*mGaSb.Eg(iT, iPoint);
	if (mInt == "lin") // interpolacja liniowa
		return ( tEgLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
	{
		if (iPoint == 'G') return ( tEgLin - mAl*mGa*mAs*(-0.127+1.310*mAl) - mAl*mGa*mSb*(-0.044+1.22*mAl) - mAl*mAs*mSb*0.8 - mGa*mAs*mSb*1.43 - mAl*mGa*mAs*mSb*0.48 );
		else if (iPoint == 'X') return ( tEgLin - mAl*mGa*mAs*0.055 /*- mAl*mGa*mSb*0.*/ - mAl*mAs*mSb*0.28 - mGa*mAs*mSb*1.2 );
		else if (iPoint == 'L') return ( tEgLin /*- mAl*mGa*mAs*0.*/ /*- mAl*mGa*mSb*0.*/ - mAl*mAs*mSb*0.28 - mGa*mAs*mSb*1.2 );
		else return 0.;
	}
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		if (iPoint == 'G')
		{
			double tEgAlGaAs = mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mAl * mGa * (-0.127+1.310*mAl);
			double tEgAlGaSb = mAl * mAlSb.Eg(iT,iPoint) + mGa * mGaSb.Eg(iT,iPoint) - mAl * mGa * (-0.044+1.22*mAl);
			double tEgAlAsSb = mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.8;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.43;
			return ( (mAl*mGa*(mAs*tEgAlGaAs+mSb*tEgAlGaSb-mAs*mSb*0.48) + (mAs*mSb*(mAl*tEgAlAsSb+mGa*tEgGaAsSb-mAl*mGa*0.48))) / (mAl*mGa+mAs*mSb) );
		}
		else if (iPoint == 'X')
		{
			double tEgAlGaAs = mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mAl * mGa * 0.055;
			double tEgAlGaSb = mAl * mAlSb.Eg(iT,iPoint) + mGa * mGaSb.Eg(iT,iPoint) /*- mAl * mGa * 0.*/;
			double tEgAlAsSb = mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.28;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.2;
			return ( (mAl*mGa*(mAs*tEgAlGaAs+mSb*tEgAlGaSb) + (mAs*mSb*(mAl*tEgAlAsSb+mGa*tEgGaAsSb))) / (mAl*mGa+mAs*mSb) );
		}		
		else if (iPoint == 'L')
		{
			double tEgAlGaAs = mAl * mAlAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) /*- mAl * mGa * 0.*/;
			double tEgAlGaSb = mAl * mAlSb.Eg(iT,iPoint) + mGa * mGaSb.Eg(iT,iPoint) /*- mAl * mGa * 0.*/;
			double tEgAlAsSb = mAs * mAlAs.Eg(iT,iPoint) + mSb * mAlSb.Eg(iT,iPoint) - mAs * mSb * 0.28;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.2;
			return ( (mAl*mGa*(mAs*tEgAlGaAs+mSb*tEgAlGaSb) + (mAs*mSb*(mAl*tEgAlAsSb+mGa*tEgGaAsSb))) / (mAl*mGa+mAs*mSb) );
		}		
		else return 0.;
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaAsSb::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double AlGaAsSb::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tDsoLin = mAl*mAs*mAlAs.Dso() + mAl*mSb*mAlSb.Dso() + mGa*mAs*mGaAs.Dso() + mGa*mSb*mGaSb.Dso();
	if (mInt == "lin") // interpolacja liniowa
		return ( tDsoLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tDsoLin /*- mAl*mGa*mAs*0.*/ - mAl*mGa*mSb*0.3 - mAl*mAs*mSb*0.15 - mGa*mAs*mSb*0.6 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tDsoAlGaAs = mAl * mAlAs.Dso() + mGa * mGaAs.Dso() /*- mAl * mGa * 0.*/;
		double tDsoAlGaSb = mAl * mAlSb.Dso() + mGa * mGaSb.Dso() - mAl * mGa * 0.3;
		double tDsoAlAsSb = mAs * mAlAs.Dso() + mSb * mAlSb.Dso() - mAs * mSb * 0.15;
		double tDsoGaAsSb = mAs * mGaAs.Dso() + mSb * mGaSb.Dso() - mAs * mSb * 0.6;
		return ( (mAl*mGa*(mAs*tDsoAlGaAs+mSb*tDsoAlGaSb) + (mAs*mSb*(mAl*tDsoAlAsSb+mGa*tDsoGaAsSb))) / (mAl*mGa+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaAsSb::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tmeLin = mAl*mAs*mAlAs.me() + mAl*mSb*mAlSb.me() + mGa*mAs*mGaAs.me() + mGa*mSb*mGaSb.me();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmeLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmeLin /*- mAl*mGa*mAs*0.*/ /*- mAl*mGa*mSb*0.*/ /*- mAl*mAs*mSb*0.*/ - mGa*mAs*mSb*0.014 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tmeAlGaAs = mAl * mAlAs.me() + mGa * mGaAs.me() /*- mAl * mGa * 0.*/;
		double tmeAlGaSb = mAl * mAlSb.me() + mGa * mGaSb.me() /*- mAl * mGa * 0.*/;
		double tmeAlAsSb = mAs * mAlAs.me() + mSb * mAlSb.me() /*- mAs * mSb * 0.*/;
		double tmeGaAsSb = mAs * mGaAs.me() + mSb * mGaSb.me() - mAs * mSb * 0.014;
		return ( (mAl*mGa*(mAs*tmeAlGaAs+mSb*tmeAlGaSb) + (mAs*mSb*(mAl*tmeAlAsSb+mGa*tmeGaAsSb))) / (mAl*mGa+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaAsSb::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tmhhLin = mAl*mAs*mAlAs.mhh() + mAl*mSb*mAlSb.mhh() + mGa*mAs*mGaAs.mhh() + mGa*mSb*mGaSb.mhh();
	return ( tmhhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tmlhLin = mAl*mAs*mAlAs.mlh() + mAl*mSb*mAlSb.mlh() + mGa*mAs*mGaAs.mlh() + mGa*mSb*mGaSb.mlh();
	return ( tmlhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tmsoLin = mAl*mAs*mAlAs.mso() + mAl*mSb*mAlSb.mso() + mGa*mAs*mGaAs.mso() + mGa*mSb*mGaSb.mso();
	return ( tmsoLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb; punktem odniesienia jest Ev=0eV dla InSb";
	}
	double tEvLin = mAl*mAs*mAlAs.Ev() + mAl*mSb*mAlSb.Ev() + mGa*mAs*mGaAs.Ev() + mGa*mSb*mGaSb.Ev();
	if (mInt == "lin") // interpolacja liniowa
		return ( tEvLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tEvLin /*- mAl*mGa*mAs*0.*/ /*- mAl*mGa*mSb*0.*/ - mAl*mAs*mSb*(-1.71) - mGa*mAs*mSb*(-1.06) );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tEvAlGaAs = mAl * mAlAs.Ev() + mGa * mGaAs.Ev() /*- mAl * mGa * 0.*/;
		double tEvAlGaSb = mAl * mAlSb.Ev() + mGa * mGaSb.Ev() /*- mAl * mGa * 0.*/;
		double tEvAlAsSb = mAs * mAlAs.Ev() + mSb * mAlSb.Ev() - mAs * mSb * (-1.71);
		double tEvGaAsSb = mAs * mGaAs.Ev() + mSb * mGaSb.Ev() - mAs * mSb * (-1.06);
		return ( (mAl*mGa*(mAs*tEvAlGaAs+mSb*tEvAlGaSb) + (mAs*mSb*(mAl*tEvAlAsSb+mGa*tEvGaAsSb))) / (mAl*mGa+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaAsSb::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tacLin = mAl*mAs*mAlAs.ac() + mAl*mSb*mAlSb.ac() + mGa*mAs*mGaAs.ac() + mGa*mSb*mGaSb.ac();
	return ( tacLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tavLin = mAl*mAs*mAlAs.av() + mAl*mSb*mAlSb.av() + mGa*mAs*mGaAs.av() + mGa*mSb*mGaSb.av();
	return ( tavLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tbLin = mAl*mAs*mAlAs.b() + mAl*mSb*mAlSb.b() + mGa*mAs*mGaAs.b() + mGa*mSb*mGaSb.b();
	return ( tbLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tdLin = mAl*mAs*mAlAs.d() + mAl*mSb*mAlSb.d() + mGa*mAs*mGaAs.d() + mGa*mSb*mGaSb.d();
	return ( tdLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tc11Lin = mAl*mAs*mAlAs.c11() + mAl*mSb*mAlSb.c11() + mGa*mAs*mGaAs.c11() + mGa*mSb*mGaSb.c11();
	return ( tc11Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tc12Lin = mAl*mAs*mAlAs.c12() + mAl*mSb*mAlSb.c12() + mGa*mAs*mGaAs.c12() + mGa*mSb*mGaSb.c12();
	return ( tc12Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tc44Lin = mAl*mAs*mAlAs.c44() + mAl*mSb*mAlSb.c44() + mGa*mAs*mGaAs.c44() + mGa*mSb*mGaSb.c44();
	return ( tc44Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	} 
	return ( 1./(mAl*mAs/mAlAs.k(iT) + mAl*mSb/mAlSb.k(iT) + mGa*mAs/mGaAs.k(iT) + mGa*mSb/mGaSb.k(iT) + mAl*mGa*0.32 + mAs*mSb*0.91) );
}
//------------------------------------------------------------------------------
double AlGaAsSb::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tcpLin = mAl*mAs*mAlAs.cp(iT) + mAl*mSb*mAlSb.cp(iT) + mGa*mAs*mGaAs.cp(iT) + mGa*mSb*mGaSb.cp(iT);
	return ( tcpLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaAsSb::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, AlSb, GaAs i GaSb";
	}
	double tgLin = mAl*mAs*mAlAs.g(iT) + mAl*mSb*mAlSb.g(iT) + mGa*mAs*mGaAs.g(iT) + mGa*mSb*mGaSb.g(iT);
	return ( tgLin ); // lin = nlin1 = nlin2
}
//---------------------------------------------------------------------------
double AlGaAsSb::nR(double iLam, double iT, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. LINNIK, A.CHRISTOU";
		ipRef->title="OPTICAL PROPERTIES OF AlGaAsSb, AlGalnP, AlGaInAs, AND GaInAsP...";
		ipRef->journal="Mat. Res. Soc. Symp. Proc. Vol. 579 pp.201, 2000";
		ipRef->comment="zaleznosc oblicza nR dla fal ok. 500-4000 nm";
	}
	if ( (iLam>=500.0) && (iLam<=4000.0) )
	{
		double Pi = cMath::pi;
		double h  = 6.62607004e-34;
		double c  = 299792458;
		double q  = 1.6021766208e-19;

		double E_photon = h*c/(q*iLam*1e-9);	//[eV]

		//Ternary alloy material parameters: bowing parameters 
		double C_AlGaAs_G = 0.37;
		double C_AlInAs_G = 0.52;
		double C_GaInAs_G = 1.4;
		double C_AlGaP_G  = 0.0;
		double C_AlInP_G  = 0.0;
		double C_GaInP_G  = 0.18;
		double C_AlGaSb_G = 0.47;
		double C_GaAsP_G  = 0.21;
		double C_AlSbAs_G = 0.0;
		double C_GaSbAs_G = 1.2;
		double C_InAsP_G  = 0.27;

		double C_AlGaAs_X = 0.245;
		double C_AlInAs_X = 0.0;
		double C_GaInAs_X = 1.4;
		double C_AlGaP_X  = 0.0;
		double C_AlInP_X  = 0.0;
		double C_GaInP_X  = 0.15;
		double C_AlGaSb_X = 0.0;
		double C_GaAsP_X  = 0.21;
		double C_AlSbAs_X = 0.0;
		double C_GaSbAs_X = 1.09;
		double C_InAsP_X  = 0.27;

		double C_AlGaAs_L = 0.055;
		double C_AlInAs_L = 0.0;
		double C_GaInAs_L = 0.72;
		double C_AlGaP_L  = 0.0;
		double C_AlInP_L  = 0.0;
		double C_GaInP_L  = 0.43;
		double C_AlGaSb_L = 0.55;
		double C_GaAsP_L  = 0.42;
		double C_AlSbAs_L = 0.0;
		double C_GaSbAs_L = 1.09;
		double C_InAsP_L  = 0.27;

		//Binary alloy material parameters: direct and indirect energy bandgaps E_G,E_X, E_L [eV], splitting energies DEL_SO [eV] A, and strength parameters A and B
		/*
		AlAs 2.95 2.16 2.36 0.28 25.30		-0.8
		AlP  3.63 2.45 4.00 0.00 18.31		 0.59
		AlSb 2.30 1.61 2.21 0.72 59.68		-9.53
		GaAs 1.43 1.91 1.73 0.34 9.29(5.3*)  7.86(10.4*)
		GaP  2.74 2.27 3.00 0.13 22.25		 0.9
		GaSb 0.72 1.05 0.76 0.74 4.05		 12.66
		InAs 0.36 1.37 1.07 0.40 4.36(6.14*) 11.65(9.15*)
		InP  1.35 2.04 1.87 0.11 8.40		 6.60
		*/
		//* Values of A and B used in calculations for AlGaInAs system.

		double E_AlAs_G = 2.95;
		double E_AlP_G  = 3.63;
		double E_AlSb_G = 2.30;
		double E_GaAs_G = 1.43;
		double E_GaP_G  = 2.74;
		double E_GaSb_G = 0.72;
		double E_InAs_G = 0.36;
		double E_InP_G  = 1.35;

		double E_AlAs_X = 2.16;
		double E_AlP_X  = 2.45;
		double E_AlSb_X = 1.61;
		double E_GaAs_X = 1.91;
		double E_GaP_X  = 2.27;
		double E_GaSb_X = 1.05;
		double E_InAs_X = 1.37;
		double E_InP_X  = 2.04;

		double E_AlAs_L = 2.36;
		double E_AlP_L  = 4.00;
		double E_AlSb_L = 2.21;
		double E_GaAs_L = 1.73;
		double E_GaP_L  = 3.00;
		double E_GaSb_L = 0.76;
		double E_InAs_L = 1.07;
		double E_InP_L  = 1.87;

		double D_SO_AlAs = 0.28;
		double D_SO_AlP  = 0.00;
		double D_SO_AlSb = 0.72;
		double D_SO_GaAs = 0.34;
		double D_SO_GaP  = 0.13;
		double D_SO_GaSb = 0.74;
		double D_SO_InAs = 0.40;
		double D_SO_InP  = 0.11;

		double A_AlAs = 25.30;
		double A_AlP  = 18.31;
		double A_AlSb = 59.68;
		double A_GaAs = 9.29;
		double A_GaP  = 22.25;
		double A_GaSb = 4.05;
		double A_InAs = 4.36;
		double A_InP  = 8.40;

		double B_AlAs = -0.80;
		double B_AlP  = 0.59;
		double B_AlSb = -9.53;
		double B_GaAs = 7.86;
		double B_GaP  = 0.90;
		double B_GaSb = 12.66;
		double B_InAs = 11.65;
		double B_InP  = 6.60;

		//dla Al_xGa_1-xAs_ySb_1-y -		A(x,y) = x*y*A_AC + x*(1-y)*A_AD +(1-x)*y*A_BC + (1-x)*(1-y)*A_BD
		double D_SO_xy	= mAl*mAs*D_SO_AlAs + mAl*(1-mAs)*D_SO_AlSb +(1-mAl)*mAs*D_SO_GaAs + (1-mAl)*(1-mAs)*D_SO_GaSb;
		double A_xy		= mAl*mAs*A_AlAs + mAl*(1-mAs)*A_AlSb +(1-mAl)*mAs*A_GaAs + (1-mAl)*(1-mAs)*A_GaSb;
		double B_xy		= mAl*mAs*B_AlAs + mAl*(1-mAs)*B_AlSb +(1-mAl)*mAs*B_GaAs + (1-mAl)*(1-mAs)*B_GaSb;

		double E_xy_G, E_xy_X, E_xy_L;

		if((mAl == 0.0) && (mAs == 0.0))		//dla GaSb
		{
			E_xy_G	= E_GaSb_G;
			E_xy_X	= E_GaSb_X;
			E_xy_L	= E_GaSb_L;
		}
		else if((mAl == 1.0) && (mAs == 1.0))		//dla AlAs
		{
			E_xy_G	= E_AlAs_G;
			E_xy_X	= E_AlAs_X;
			E_xy_L	= E_AlAs_L;
		}
		else if((mAl == 1.0) && (mAs == 0.0))		//dla AlSb
		{
			E_xy_G	= E_AlSb_G;
			E_xy_X	= E_AlSb_X;
			E_xy_L	= E_AlSb_L;
		}
		else if((mAl == 0.0) && (mAs == 1.0))		//dla GaAs
		{
			E_xy_G	= E_GaAs_G;
			E_xy_X	= E_GaAs_X;
			E_xy_L	= E_GaAs_L;
		}
		else if((mAl == 0.0) && (mAs != 0.0))	//dla GaSbAs
		{
			E_xy_G	= mAs*E_GaAs_G + (1-mAs)*E_GaSb_G  - mAs*(1-mAs)*C_GaSbAs_G;
			E_xy_X	= mAs*E_GaAs_X + (1-mAs)*E_GaSb_X  - mAs*(1-mAs)*C_GaSbAs_X;
			E_xy_L	= mAs*E_GaAs_L + (1-mAs)*E_GaSb_L  - mAs*(1-mAs)*C_GaSbAs_L;
		}
		else if((mAl != 0.0) && (mAs == 0.0))	//dla AlGaSb
		{
			E_xy_G	= mAl*E_AlSb_G  + (1-mAl)*E_GaSb_G  - mAl*(1-mAl)*C_AlGaSb_G;
			E_xy_X	= mAl*E_AlSb_X  + (1-mAl)*E_GaSb_X  - mAl*(1-mAl)*C_AlGaSb_X;
			E_xy_L	= mAl*E_AlSb_L  + (1-mAl)*E_GaSb_L  - mAl*(1-mAl)*C_AlGaSb_L;
		}
		else if((mAl == 1.0) && (mAs != 0.0))	//dla AlSbAs
		{
			E_xy_G	= mAs*E_AlAs_G  + (1-mAs)*E_AlSb_G  - mAs*(1-mAs)*C_AlSbAs_G;
			E_xy_X	= mAs*E_AlAs_X  + (1-mAs)*E_AlSb_X  - mAs*(1-mAs)*C_AlSbAs_X;
			E_xy_L	= mAs*E_AlAs_L  + (1-mAs)*E_AlSb_L  - mAs*(1-mAs)*C_AlSbAs_L;
		}
		else if((mAl != 0.0) && (mAs == 1.0))	//dla AlGaAs
		{
			E_xy_G	= mAl*E_AlAs_G  + (1-mAl)*E_GaAs_G  - mAl*(1-mAl)*C_AlGaAs_G;
			E_xy_X	= mAl*E_AlAs_X  + (1-mAl)*E_GaAs_X  - mAl*(1-mAl)*C_AlGaAs_X;
			E_xy_L	= mAl*E_AlAs_L  + (1-mAl)*E_GaAs_L  - mAl*(1-mAl)*C_AlGaAs_L;
		}
		else
		{
			//dla Al_xGa_1-xAs_ySb_1-y -	Eg(x,y) = (x*(1-x)*((1-y)*E_ABD_x + y*E_ABC_x) + y*(1-y)*((1-x)*E_BCD_y + x*E_ACD_y)) / (x*(1-x)+y*(1-y))
			//								E_ABC_x = x*E_AB + (1-x)*E_AC - x*(1-x)*C_ABC
			//								itd.
			double E_AlGaSb_G	= mAl*E_AlSb_G + (1-mAl)*E_GaSb_G - mAl*(1-mAl)*C_AlGaSb_G;
			double E_AlGaAs_G	= mAl*E_AlAs_G + (1-mAl)*E_GaAs_G - mAl*(1-mAl)*C_AlGaAs_G;
			double E_AlSbAs_G	= mAs*E_AlAs_G + (1-mAs)*E_AlSb_G - mAs*(1-mAs)*C_AlSbAs_G;
			double E_GaSbAs_G	= mAs*E_GaAs_G + (1-mAs)*E_GaSb_G - mAs*(1-mAs)*C_GaSbAs_G;

			E_xy_G	= (mAl*(1-mAl)*((1-mAs)*E_AlGaSb_G + mAs*E_AlGaAs_G) + mAs*(1-mAs)*((1-mAl)*E_AlSbAs_G + mAl*E_GaSbAs_G)) / (mAl*(1-mAl)+mAs*(1-mAs));


			double E_AlGaSb_X	= mAl*E_AlSb_X + (1-mAl)*E_GaSb_X - mAl*(1-mAl)*C_AlGaSb_X;
			double E_AlGaAs_X	= mAl*E_AlAs_X + (1-mAl)*E_GaAs_X - mAl*(1-mAl)*C_AlGaAs_X;
			double E_AlSbAs_X	= mAs*E_AlAs_X + (1-mAs)*E_AlSb_X - mAs*(1-mAs)*C_AlSbAs_X;
			double E_GaSbAs_X	= mAs*E_GaAs_X + (1-mAs)*E_GaSb_X - mAs*(1-mAs)*C_GaSbAs_X;

			E_xy_X	= (mAl*(1-mAl)*((1-mAs)*E_AlGaSb_X + mAs*E_AlGaAs_X) + mAs*(1-mAs)*((1-mAl)*E_AlSbAs_X + mAl*E_GaSbAs_X)) / (mAl*(1-mAl)+mAs*(1-mAs));


			double E_AlGaSb_L	= mAl*E_AlSb_L + (1-mAl)*E_GaSb_L - mAl*(1-mAl)*C_AlGaSb_L;
			double E_AlGaAs_L	= mAl*E_AlAs_L + (1-mAl)*E_GaAs_L - mAl*(1-mAl)*C_AlGaAs_L;
			double E_AlSbAs_L	= mAs*E_AlAs_L + (1-mAs)*E_AlSb_L - mAs*(1-mAs)*C_AlSbAs_L;
			double E_GaSbAs_L	= mAs*E_GaAs_L + (1-mAs)*E_GaSb_L - mAs*(1-mAs)*C_GaSbAs_L;

			E_xy_L	= (mAl*(1-mAl)*((1-mAs)*E_AlGaSb_L + mAs*E_AlGaAs_L) + mAs*(1-mAs)*((1-mAl)*E_AlSbAs_L + mAl*E_GaSbAs_L)) / (mAl*(1-mAl)+mAs*(1-mAs));
		}

		//wybieramy najmniejsze E_xy
		double E_xy		= E_xy_G;
		if(E_xy_X < E_xy) E_xy = E_xy_X;
		if(E_xy_L < E_xy) E_xy = E_xy_L;

		double Ksi0  = E_photon/E_xy;
		double KsiSO = E_photon/(E_xy+D_SO_xy);

		double H0 = 1.0;
		if((1-Ksi0)<0)	H0 = 0.0;
		else			H0 = sqrt(1 - Ksi0);

		double HSO = 1.0;
		if((1-KsiSO)<0) HSO = 0.0;
		else			HSO = sqrt(1 - KsiSO);

		double F_Ksi0 = (2 - sqrt(1 + Ksi0) - H0)/(Ksi0*Ksi0);
		double F_KsiSO = (2 - sqrt(1 + KsiSO) - HSO)/(KsiSO*KsiSO);

		double nR = sqrt(A_xy*(F_Ksi0 + 0.5*pow(E_xy/(E_xy+D_SO_xy), 1.5)*F_KsiSO) + B_xy);

		//zaleznosc temperaturowa na bazie pracy:
		//brak
		
		double dn_dT = 0.0;	//1/K

		return nR + dn_dT*(iT-300);
	}
	else
		return 0.;
}
//---------------------------------------------------------------------------
void AlGaAsSb::setInt(std::string iInt)
{
	mInt = iInt;
}
//---------------------------------------------------------------------------
AlGaAsSb::AlGaAsSb(double iAl,double iAs)
{
	mAl = iAl;
	mGa = 1.-iAl;
	mAs = iAs;
	mSb = 1.-mAs;
	mInt = "nlin1";
}
//---------------------------------------------------------------------------
AlGaAsSb::AlGaAsSb()
{  
}
//---------------------------------------------------------------------------
AlGaAsSb::~AlGaAsSb()
{
}
