#include "pch.h"

#include "AlGaInAs.h"
//------------------------------------------------------------------------------
double AlGaInAs::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	return ( mAl * mAlAs.a(iT) + mGa * mGaAs.a(iT) + mIn * mInAs.a(iT) );
}
//------------------------------------------------------------------------------
double AlGaInAs::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tEgLin = mAl * mAlAs.Eg(iT, iPoint) + mGa * mGaAs.Eg(iT, iPoint) + mIn * mInAs.Eg(iT, iPoint);
	if (mInt == "lin") // interpolacja liniowa
		return ( tEgLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
	{
		if (iPoint == 'G') return ( tEgLin - mAl*mGa*(-0.127+1.310*mAl) - mAl*mIn*0.70 - mGa*mIn*0.477 - mAl*mGa*mIn*0.22 ); 
		else if (iPoint == 'X') return ( tEgLin - mAl*mGa*0.055 /*- mAl*mIn*0.*/ - mGa*mIn*1.4 );
		else if (iPoint == 'L') return ( tEgLin /*- mAl*mGa*0.*/ /*- mAl*mIn*0.*/ - mGa*mIn*0.33 );
		else return 0.;
	}
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		if (iPoint == 'G')
		{
			double tEgAlInAs = tU * mAlAs.Eg(iT,iPoint) + (1.-tU) * mInAs.Eg(iT,iPoint) - tU * (1.-tU) * 0.70;
			double tEgAlGaAs = tV * mGaAs.Eg(iT,iPoint) + (1.-tV) * mAlAs.Eg(iT,iPoint) - tV * (1.-tV) * (-0.127+1.310*(1.-tV));
			double tEgInGaAs = tZ * mInAs.Eg(iT,iPoint) + (1.-tZ) * mGaAs.Eg(iT,iPoint) - tZ * (1.-tZ) * 0.477;
			return ( (mIn*mAl*tEgAlInAs + mAl*mGa*tEgAlGaAs + mGa*mIn*tEgInGaAs - mAl*mGa*mIn*0.22)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}
		else if (iPoint == 'X')
		{
			double tEgAlInAs = tU * mAlAs.Eg(iT,iPoint) + (1.-tU) * mInAs.Eg(iT,iPoint) /*- tU * (1.-tU) * 0.*/;
			double tEgAlGaAs = tV * mGaAs.Eg(iT,iPoint) + (1.-tV) * mAlAs.Eg(iT,iPoint) - tV * (1.-tV) * 0.055;
			double tEgInGaAs = tZ * mInAs.Eg(iT,iPoint) + (1.-tZ) * mGaAs.Eg(iT,iPoint) - tZ * (1.-tZ) * 1.4;
			return ( (mIn*mAl*tEgAlInAs + mAl*mGa*tEgAlGaAs + mGa*mIn*tEgInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}		
		else if (iPoint == 'L')
		{
			double tEgAlInAs = tU * mAlAs.Eg(iT,iPoint) + (1.-tU) * mInAs.Eg(iT,iPoint) /*- tU * (1.-tU) * 0.*/;
			double tEgAlGaAs = tV * mGaAs.Eg(iT,iPoint) + (1.-tV) * mAlAs.Eg(iT,iPoint) /*- tV * (1.-tV) * 0.*/;
			double tEgInGaAs = tZ * mInAs.Eg(iT,iPoint) + (1.-tZ) * mGaAs.Eg(iT,iPoint) - tZ * (1.-tZ) * 0.33;
			return ( (mIn*mAl*tEgAlInAs + mAl*mGa*tEgAlGaAs + mGa*mIn*tEgInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}		
		else return 0.;
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInAs::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double AlGaInAs::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tDsoLin = mAl * mAlAs.Dso() + mGa * mGaAs.Dso() + mIn * mInAs.Dso();
	if (mInt == "lin") // interpolacja liniowa
		return ( tDsoLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tDsoLin /*- mAl*mGa*0.*/ - mAl*mIn*0.15 - mGa*mIn*0.15 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tDsoAlInAs = tU * mAlAs.Dso() + (1.-tU) * mInAs.Dso() - tU * (1.-tU) * 0.15;
		double tDsoAlGaAs = tV * mGaAs.Dso() + (1.-tV) * mAlAs.Dso() /*- tV * (1.-tV) * 0.*/;
		double tDsoInGaAs = tZ * mInAs.Dso() + (1.-tZ) * mGaAs.Dso() - tZ * (1.-tZ) * 0.15;
		return ( (mIn*mAl*tDsoAlInAs + mAl*mGa*tDsoAlGaAs + mGa*mIn*tDsoInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );	
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInAs::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs"; 
	}
	double tmeLin = mAl * mAlAs.me() + mGa * mGaAs.me() + mIn * mInAs.me();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmeLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmeLin /*- mAl*mGa*0.*/ - mAl*mIn*0.012 - mGa*mIn*0.008 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tmeAlInAs = tU * mAlAs.me() + (1.-tU) * mInAs.me() - tU * (1.-tU) * 0.012;
		double tmeAlGaAs = tV * mGaAs.me() + (1.-tV) * mAlAs.me() /*- tV * (1.-tV) * 0.*/;
		double tmeInGaAs = tZ * mInAs.me() + (1.-tZ) * mGaAs.me() - tZ * (1.-tZ) * 0.008;
		return ( (mIn*mAl*tmeAlInAs + mAl*mGa*tmeAlGaAs + mGa*mIn*tmeInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );	
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInAs::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tmhhLin = mAl * mAlAs.mhh() + mGa * mGaAs.mhh() + mIn * mInAs.mhh();
	return ( tmhhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tmlhLin = mAl * mAlAs.mlh() + mGa * mGaAs.mlh() + mIn * mInAs.mlh();
	return ( tmlhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tmsoLin = mAl * mAlAs.mso() + mGa * mGaAs.mso() + mIn * mInAs.mso();
	return ( tmsoLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double AlGaInAs::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs; punktem odniesienia jest Ev=0eV dla InSb";
	}
	double tEvLin = mAl * mAlAs.Ev() + mGa * mGaAs.Ev() + mIn * mInAs.Ev();
	if (mInt == "lin") // interpolacja liniowa
		return ( tEvLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tEvLin /*- mAl*mGa*0.*/ - mAl*mIn*(-0.64) - mGa*mIn*(-0.38) );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tEvAlInAs = tU * mAlAs.Ev() + (1.-tU) * mInAs.Ev() - tU * (1.-tU) * (-0.64);
		double tEvAlGaAs = tV * mGaAs.Ev() + (1.-tV) * mAlAs.Ev() /*- tV * (1.-tV) * 0.*/;
		double tEvInGaAs = tZ * mInAs.Ev() + (1.-tZ) * mGaAs.Ev() - tZ * (1.-tZ) * (-0.38);
		return ( (mIn*mAl*tEvAlInAs + mAl*mGa*tEvAlGaAs + mGa*mIn*tEvInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );	
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInAs::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double AlGaInAs::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tacLin = mAl * mAlAs.ac() + mGa * mGaAs.ac() + mIn * mInAs.ac();
	if (mInt == "lin") // interpolacja liniowa
		return ( tacLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tacLin /*- mAl*mGa*0.*/ - mAl*mIn*(-1.4) - mGa*mIn*2.61 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tEvAlInAs = tU * mAlAs.Ev() + (1.-tU) * mInAs.Ev() - tU * (1.-tU) * (-1.4);
		double tEvAlGaAs = tV * mGaAs.Ev() + (1.-tV) * mAlAs.Ev() /*- tV * (1.-tV) * 0.*/;
		double tEvInGaAs = tZ * mInAs.Ev() + (1.-tZ) * mGaAs.Ev() - tZ * (1.-tZ) * 2.61;
		return ( (mIn*mAl*tEvAlInAs + mAl*mGa*tEvAlGaAs + mGa*mIn*tEvInGaAs)/(mIn*mAl + mAl*mGa + mGa*mIn) );	
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInAs::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tavLin = mAl * mAlAs.av() + mGa * mGaAs.av() + mIn * mInAs.av();
	return ( tavLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tbLin = mAl * mAlAs.b() + mGa * mGaAs.b() + mIn * mInAs.b();
	return ( tbLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tdLin = mAl * mAlAs.d() + mGa * mGaAs.d() + mIn * mInAs.d();
	return ( tdLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tc11Lin = mAl * mAlAs.c11() + mGa * mGaAs.c11() + mIn * mInAs.c11();
	return ( tc11Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tc12Lin = mAl * mAlAs.c12() + mGa * mGaAs.c12() + mIn * mInAs.c12();
	return ( tc12Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tc44Lin = mAl * mAlAs.c44() + mGa * mGaAs.c44() + mIn * mInAs.c44();
	return ( tc44Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double AlGaInAs::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double AlGaInAs::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double AlGaInAs::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	} 
	return ( 1./(mAl/mAlAs.k(iT) + mGa/mGaAs.k(iT) + mIn/mInAs.k(iT) + mAl*mGa*0.32 + mAl*mIn*0.15 + mGa*mIn*0.72) );
}
//------------------------------------------------------------------------------
double AlGaInAs::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tcpLin = mAl * mAlAs.cp(iT) + mGa * mGaAs.cp(iT) + mIn * mInAs.cp(iT);
	return ( tcpLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlAs, GaAs i InAs";
	}
	double tgLin = mAl * mAlAs.g(iT) + mGa * mGaAs.g(iT) + mIn * mInAs.g(iT);
	return ( tgLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInAs::nR(double iLam, double iT, reference *ipRef) 
{
	//stara zaleznosc/////////////////////////////////////////////////////////////////////////////
	/*
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="H. Dejun and K.T. Chan";
		ipRef->title="Refractive index of AlInGaAs layers in the transparent wavelength regio";
		ipRef->journal="-";
		ipRef->comment="zaleznosc poprawna dla 1130-1550 nm gdy: Al=0.25-0.33 oraz In ok.53";
	}
	if ( (iLam>=1130.0) && (iLam<=1550.0) )
	{
		double E0 = 1.424+1.455*mAl + 0.191*mAl*mAl - 1.614*mIn + 0.55*mIn*mIn + 0.043*mAl*mIn;
		double Dso = 0.3*mAl + 0.43*mIn + 0.34*mGa;
		double X = cFunc::LamtoE(iLam) / E0;
		double Xso = cFunc::LamtoE(iLam) / (E0+Dso);
		double fX = (2.-sqrt(1.+X)-sqrt(1.-X))/X/X;
		double fXso = (2.-sqrt(1.+Xso)-sqrt(1.-Xso))/Xso/Xso;
		double A0 = 2.9715 + 11.6407*mAl + 1.5749*mIn;
		double B0 = 8.7279 - 10.2892*mAl + 4.62578*mIn;
		double eps1 = A0*(fX+0.5*pow(E0/(E0+Dso),1.5)*fXso)+B0;
		return ( sqrt(eps1) );    
	}
	else
		return 0.;
	*/
	////////////////////////////////////////////////////////////////////////////////////////////////

	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. LINNIK, A.CHRISTOU";
		ipRef->title="OPTICAL PROPERTIES OF AlGaAsSb, AlGalnP, AlGaInAs, AND GaInAsP...";
		ipRef->journal="Mat. Res. Soc. Symp. Proc. Vol. 579 pp.201, 2000";
		ipRef->comment="zaleznosc oblicza nR dla fal 850-4000 nm";
	}
	if ( (iLam>=850.0) && (iLam<=4000.0) )
	{
		double Pi = cMath::pi;
		double h  = 6.62607004e-34;
		double c  = 299792458;
		double q  = 1.6021766208e-19;

		double E_photon = h*c/(q*iLam*1e-9);	//[eV]

		//Ternary alloy material parameters: bowing parameters 
		double C_AlGaAs_G = 0.37;
		double C_AlInAs_G = 0.52;
		double C_GaInAs_G = 1.4;
		double C_AlGaP_G  = 0.0;
		double C_AlInP_G  = 0.0;
		double C_GaInP_G  = 0.18;
		double C_AlGaSb_G = 0.47;
		double C_GaAsP_G  = 0.21;
		double C_AlSbAs_G = 0.0;
		double C_GaSbAs_G = 1.2;
		double C_InAsP_G  = 0.27;

		double C_AlGaAs_X = 0.245;
		double C_AlInAs_X = 0.0;
		double C_GaInAs_X = 1.4;
		double C_AlGaP_X  = 0.0;
		double C_AlInP_X  = 0.0;
		double C_GaInP_X  = 0.15;
		double C_AlGaSb_X = 0.0;
		double C_GaAsP_X  = 0.21;
		double C_AlSbAs_X = 0.0;
		double C_GaSbAs_X = 1.09;
		double C_InAsP_X  = 0.27;

		double C_AlGaAs_L = 0.055;
		double C_AlInAs_L = 0.0;
		double C_GaInAs_L = 0.72;
		double C_AlGaP_L  = 0.0;
		double C_AlInP_L  = 0.0;
		double C_GaInP_L  = 0.43;
		double C_AlGaSb_L = 0.55;
		double C_GaAsP_L  = 0.42;
		double C_AlSbAs_L = 0.0;
		double C_GaSbAs_L = 1.09;
		double C_InAsP_L  = 0.27;

		//Binary alloy material parameters: direct and indirect energy bandgaps E_G,E_X, E_L [eV], splitting energies DEL_SO [eV] A, and strength parameters A and B
		/*
		AlAs 2.95 2.16 2.36 0.28 25.30		-0.8
		AlP  3.63 2.45 4.00 0.00 18.31		 0.59
		AlSb 2.30 1.61 2.21 0.72 59.68		-9.53
		GaAs 1.43 1.91 1.73 0.34 9.29(5.3*)  7.86(10.4*)
		GaP  2.74 2.27 3.00 0.13 22.25		 0.9
		GaSb 0.72 1.05 0.76 0.74 4.05		 12.66
		InAs 0.36 1.37 1.07 0.40 4.36(6.14*) 11.65(9.15*)
		InP  1.35 2.04 1.87 0.11 8.40		 6.60
		*/
		//* Values of A and B used in calculations for AlGaInAs system.

		double E_AlAs_G = 2.95;
		double E_AlP_G  = 3.63;
		double E_AlSb_G = 2.30;
		double E_GaAs_G = 1.43;
		double E_GaP_G  = 2.74;
		double E_GaSb_G = 0.72;
		double E_InAs_G = 0.36;
		double E_InP_G  = 1.35;

		double E_AlAs_X = 2.16;
		double E_AlP_X  = 2.45;
		double E_AlSb_X = 1.61;
		double E_GaAs_X = 1.91;
		double E_GaP_X  = 2.27;
		double E_GaSb_X = 1.05;
		double E_InAs_X = 1.37;
		double E_InP_X  = 2.04;

		double E_AlAs_L = 2.36;
		double E_AlP_L  = 4.00;
		double E_AlSb_L = 2.21;
		double E_GaAs_L = 1.73;
		double E_GaP_L  = 3.00;
		double E_GaSb_L = 0.76;
		double E_InAs_L = 1.07;
		double E_InP_L  = 1.87;

		double D_SO_AlAs = 0.28;
		double D_SO_AlP  = 0.00;
		double D_SO_AlSb = 0.72;
		double D_SO_GaAs = 0.34;
		double D_SO_GaP  = 0.13;
		double D_SO_GaSb = 0.74;
		double D_SO_InAs = 0.40;
		double D_SO_InP  = 0.11;

		double A_AlAs = 25.30;
		double A_AlP  = 18.31;
		double A_AlSb = 59.68;
		double A_GaAs = 5.30;
		double A_GaP  = 22.25;
		double A_GaSb = 4.05;
		double A_InAs = 6.14;
		double A_InP  = 8.40;

		double B_AlAs = -0.80;
		double B_AlP  = 0.59;
		double B_AlSb = -9.53;
		double B_GaAs = 10.4;
		double B_GaP  = 0.90;
		double B_GaSb = 12.66;
		double B_InAs = 9.15;
		double B_InP  = 6.60;

		//dla Al_xGa_yIn_1-x-yAs -		A(x,y) = x*A_AD +y*A_BD +(1-x-y)*A_CD
		double D_SO_xy	= mAl*D_SO_AlAs + mGa*D_SO_GaAs +(1-mAl-mGa)*D_SO_InAs;
		double A_xy		= mAl*A_AlAs + mGa*A_GaAs +(1-mAl-mGa)*A_InAs;
		double B_xy		= mAl*B_AlAs + mGa*B_GaAs +(1-mAl-mGa)*B_InAs;

		bool   blad = false;
		double E_xy_G, E_xy_X, E_xy_L;

		if((mAl+mGa) > 1.0)		//nie ma takich zwiazkow
		{
			blad = true;
		}
		else if((mAl == 0.0) && (mGa == 0.0))	//dla InAs
		{
			E_xy_G	= E_InAs_G;
			E_xy_X	= E_InAs_X;
			E_xy_L	= E_InAs_L;
		}
		else if((mAl == 1.0) && (mGa == 0.0))	//dla AlAs
		{
			E_xy_G	= E_AlAs_G;
			E_xy_X	= E_AlAs_X;
			E_xy_L	= E_AlAs_L;
		}
		else if((mAl == 0.0) && (mGa == 1.0))	//dla GaAs
		{
			E_xy_G	= E_GaAs_G;
			E_xy_X	= E_GaAs_X;
			E_xy_L	= E_GaAs_L;
		}
		else if((mAl == 0.0) && (mGa != 0.0))	//dla GaInAs
		{
			E_xy_G	= mGa*E_GaAs_G + (1-mGa)*E_InAs_G  - mGa*(1-mGa)*C_GaInAs_G;
			E_xy_X	= mGa*E_GaAs_X + (1-mGa)*E_InAs_X  - mGa*(1-mGa)*C_GaInAs_X;
			E_xy_L	= mGa*E_GaAs_L + (1-mGa)*E_InAs_L  - mGa*(1-mGa)*C_GaInAs_L;
		}
		else if((mAl != 0.0) && (mGa == 0.0))	//dla AlInAs
		{
			E_xy_G	= mAl*E_AlAs_G + (1-mAl)*E_InAs_G - mAl*(1-mAl)*C_AlInAs_G;
			E_xy_X	= mAl*E_AlAs_X + (1-mAl)*E_InAs_X - mAl*(1-mAl)*C_AlInAs_X;
			E_xy_L	= mAl*E_AlAs_L + (1-mAl)*E_InAs_L - mAl*(1-mAl)*C_AlInAs_L;
		}
		else if((mAl+mGa) == 1.0)				//dla AlGaAs
		{
			E_xy_G	= mAl*E_AlAs_G + (1-mAl)*E_GaAs_G - mAl*(1-mAl)*C_AlGaAs_G;
			E_xy_X	= mAl*E_AlAs_X + (1-mAl)*E_GaAs_X - mAl*(1-mAl)*C_AlGaAs_X;
			E_xy_L	= mAl*E_AlAs_L + (1-mAl)*E_GaAs_L - mAl*(1-mAl)*C_AlGaAs_L;
		}
		else
		{
			//dla Al_xGa_yIn_1-x-yAs -		Eg(x,y) = (x*y*E_ABD(u) + y*(1-x-y)*E_BCD(v) + x*(1-x-y)*E_ACD(w)) / (x*y+x*(1-x-y)+y*(1-x-y))
			//								E_ABC_u = u*E_AB + (1-u)*E_AC - u*(1-u)*C_ABC
			//								u = (1-x+y)/2	v = (2-x-2y)/2		w = (2-2x-y)/2
			//								itd.
			double u = (1-mAl+mGa)/2;	
			double v = (2-mAl-2*mGa)/2;
			double w = (2-2*mAl-mGa)/2;

			double E_AlGaAs_G	= u*E_AlAs_G  + (1-u)*E_GaAs_G  - u*(1-u)*C_AlGaAs_G;
			double E_GaInAs_G	= v*E_GaAs_G  + (1-v)*E_InAs_G  - v*(1-v)*C_GaInAs_G;
			double E_AlInAs_G	= w*E_AlAs_G  + (1-w)*E_InAs_G  - w*(1-w)*C_AlInAs_G;

			//double E_AlGaAs_G	= (1-u)*E_AlAs_G + u*E_GaAs_G - u*(1-u)*C_AlGaAs_G;
			//double E_GaInAs_G	= (1-v)*E_GaAs_G + v*E_InAs_G - v*(1-v)*C_GaInAs_G;
			//double E_AlInAs_G	= (1-w)*E_AlAs_G + w*E_InAs_G - w*(1-w)*C_AlInAs_G;

			E_xy_G	= (mAl*mGa*E_AlGaAs_G + mGa*(1-mAl-mGa)*E_GaInAs_G + mAl*(1-mAl-mGa)*E_AlInAs_G) / (mAl*mGa+mAl*(1-mAl-mGa)+mGa*(1-mAl-mGa));


			double E_AlGaAs_X	= u*E_AlAs_X  + (1-u)*E_GaAs_X  - u*(1-u)*C_AlGaAs_X;
			double E_GaInAs_X	= v*E_GaAs_X  + (1-v)*E_InAs_X  - v*(1-v)*C_GaInAs_X;
			double E_AlInAs_X	= w*E_AlAs_X  + (1-w)*E_InAs_X  - w*(1-w)*C_AlInAs_X;

			E_xy_X	= (mAl*mGa*E_AlGaAs_X + mGa*(1-mAl-mGa)*E_GaInAs_X + mAl*(1-mAl-mGa)*E_AlInAs_X) / (mAl*mGa+mAl*(1-mAl-mGa)+mGa*(1-mAl-mGa));


			double E_AlGaAs_L	= u*E_AlAs_L  + (1-u)*E_GaAs_L  - u*(1-u)*C_AlGaAs_L;
			double E_GaInAs_L	= v*E_GaAs_L  + (1-v)*E_InAs_L  - v*(1-v)*C_GaInAs_L;
			double E_AlInAs_L	= w*E_AlAs_L  + (1-w)*E_InAs_L  - w*(1-w)*C_AlInAs_L;

			E_xy_L	= (mAl*mGa*E_AlGaAs_L + mGa*(1-mAl-mGa)*E_GaInAs_L + mAl*(1-mAl-mGa)*E_AlInAs_L) / (mAl*mGa+mAl*(1-mAl-mGa)+mGa*(1-mAl-mGa));
		}

		double nR = 0.0;
		if(!blad)
		{
			//wybieramy najmniejsze E_xy
			double E_xy		= E_xy_G;
			if(E_xy_X < E_xy) E_xy = E_xy_X;
			if(E_xy_L < E_xy) E_xy = E_xy_L;

			double Ksi0  = E_photon/E_xy;
			double KsiSO = E_photon/(E_xy+D_SO_xy);

			double H0 = 1.0;
			if((1-Ksi0)<0)	H0 = 0.0;
			else			H0 = sqrt(1 - Ksi0);

			double HSO = 1.0;
			if((1-KsiSO)<0) HSO = 0.0;
			else			HSO = sqrt(1 - KsiSO);

			double F_Ksi0 = (2 - sqrt(1 + Ksi0) - H0)/(Ksi0*Ksi0);
			double F_KsiSO = (2 - sqrt(1 + KsiSO) - HSO)/(KsiSO*KsiSO);

			nR = sqrt(A_xy*(F_Ksi0 + 0.5*pow(E_xy/(E_xy+D_SO_xy), 1.5)*F_KsiSO) + B_xy);
			

			//zaleznosc temperaturowa na bazie pracy:
			//Abi Waqas, Daniele Melati, Zarlish Mushtaq, B.S. Chowdhry, Andrea Melloni;
			//An Improved Model to Predict the Temperature Dependence of Refractive Index of InP-based Compounds
			//Wireless Pers Commun. 2016; DOI 10.1007/s11277-016-3913-5
			//double dn_dT = 1e-4;	//1/K

			//if(iLam < 1250) dn_dT *= 2.3;
			//else if(iLam < 1350) dn_dT *= 2.1;
			//else if(iLam < 1450) dn_dT *= 1.97;
			//else dn_dT *= 1.9;

			//nR += dn_dT*(iT-300);
		}
		return nR;
	}
	else
		return 0.;
}
//---------------------------------------------------------------------------
void AlGaInAs::setInt(std::string iInt)
{
	mInt = iInt;
}
//---------------------------------------------------------------------------
AlGaInAs::AlGaInAs(double iAl,double iGa)
{
	mAl = iAl;
	mGa = iGa;
	mIn = 1.-iAl-iGa;
	mInt = "nlin1";
}
//---------------------------------------------------------------------------
AlGaInAs::AlGaInAs()
{  
}
//---------------------------------------------------------------------------
AlGaInAs::~AlGaInAs()
{
}
