#include "pch.h"

#include "AlGaInP.h"
//------------------------------------------------------------------------------
double AlGaInP::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	return ( mAl * mAlP.a(iT) + mGa * mGaP.a(iT) + mIn * mInP.a(iT) );
}
//------------------------------------------------------------------------------
double AlGaInP::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tEgLin = mAl * mAlP.Eg(iT, iPoint) + mGa * mGaP.Eg(iT, iPoint) + mIn * mInP.Eg(iT, iPoint);
	if (mInt == "lin") // interpolacja liniowa
		return ( tEgLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
	{
		if (iPoint == 'G') return ( tEgLin /*- mAl*mGa*0.*/ - mAl*mIn*(-0.48) - mGa*mIn*0.65 - mAl*mGa*mIn*0.18 ); 
		else if (iPoint == 'X') return ( tEgLin - mAl*mGa*0.13 - mAl*mIn*0.38 - mGa*mIn*0.20 );
		else if (iPoint == 'L') return ( tEgLin /*- mAl*mGa*0.*/ /*- mAl*mIn*0.*/ - mGa*mIn*1.03 );
		else return 0.;
	}
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		if (iPoint == 'G')
		{
			double tEgAlInP = tU * mAlP.Eg(iT,iPoint) + (1.-tU) * mInP.Eg(iT,iPoint) - tU * (1.-tU) * (-0.48);
			double tEgAlGaP = tV * mGaP.Eg(iT,iPoint) + (1.-tV) * mAlP.Eg(iT,iPoint) /*- tV * (1.-tV) * 0.*/;
			double tEgInGaP = tZ * mInP.Eg(iT,iPoint) + (1.-tZ) * mGaP.Eg(iT,iPoint) - tZ * (1.-tZ) * 0.65;
			return ( (mIn*mAl*tEgAlInP + mAl*mGa*tEgAlGaP + mGa*mIn*tEgInGaP - mAl*mGa*mIn*0.22)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}
		else if (iPoint == 'X')
		{
			double tEgAlInP = tU * mAlP.Eg(iT,iPoint) + (1.-tU) * mInP.Eg(iT,iPoint) - tU * (1.-tU) * 0.38;
			double tEgAlGaP = tV * mGaP.Eg(iT,iPoint) + (1.-tV) * mAlP.Eg(iT,iPoint) - tV * (1.-tV) * 0.13;
			double tEgInGaP = tZ * mInP.Eg(iT,iPoint) + (1.-tZ) * mGaP.Eg(iT,iPoint) - tZ * (1.-tZ) * 0.20;
			return ( (mIn*mAl*tEgAlInP + mAl*mGa*tEgAlGaP + mGa*mIn*tEgInGaP)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}		
		else if (iPoint == 'L')
		{
			double tEgAlInP = tU * mAlP.Eg(iT,iPoint) + (1.-tU) * mInP.Eg(iT,iPoint) /*- tU * (1.-tU) * 0.*/;
			double tEgAlGaP = tV * mGaP.Eg(iT,iPoint) + (1.-tV) * mAlP.Eg(iT,iPoint) /*- tV * (1.-tV) * 0.*/;
			double tEgInGaP = tZ * mInP.Eg(iT,iPoint) + (1.-tZ) * mGaP.Eg(iT,iPoint) - tZ * (1.-tZ) * 1.03;
			return ( (mIn*mAl*tEgAlInP + mAl*mGa*tEgAlGaP + mGa*mIn*tEgInGaP -mAl*mGa*mIn*0.18)/(mIn*mAl + mAl*mGa + mGa*mIn) );
		}		
		else return 0.;
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInP::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double AlGaInP::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tDsoLin = mAl * mAlP.Dso() + mGa * mGaP.Dso() + mIn * mInP.Dso();
	if (mInt == "lin") // interpolacja liniowa
		return ( tDsoLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tDsoLin /*- mAl*mGa*0.*/ - mAl*mIn*(-0.19) /*- mGa*mIn*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tDsoAlInP = tU * mAlP.Dso() + (1.-tU) * mInP.Dso() - tU * (1.-tU) * (-0.19);
		double tDsoAlGaP = tV * mGaP.Dso() + (1.-tV) * mAlP.Dso() /*- tV * (1.-tV) * 0.*/;
		double tDsoInGaP = tZ * mInP.Dso() + (1.-tZ) * mGaP.Dso() /*- tZ * (1.-tZ) * 0.*/;
		return ( (mIn*mAl*tDsoAlInP + mAl*mGa*tDsoAlGaP + mGa*mIn*tDsoInGaP)/(mIn*mAl + mAl*mGa + mGa*mIn) );		
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInP::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP"; 
	}
	double tmeLin = mAl * mAlP.me() + mGa * mGaP.me() + mIn * mInP.me();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmeLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmeLin /*- mAl*mGa*0.*/ /*- mAl*mIn*0.*/ - mGa*mIn*0.01854 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tU = 0.5*(1.-mIn+mAl);
		double tV = 0.5*(1.-mAl+mGa);
		double tZ = 0.5*(1.-mGa+mIn);		
		double tmeAlInP = tU * mAlP.me() + (1.-tU) * mInP.me() /*- tU * (1.-tU) * 0.*/;
		double tmeAlGaP = tV * mGaP.me() + (1.-tV) * mAlP.me() /*- tV * (1.-tV) * 0.*/;
		double tmeInGaP = tZ * mInP.me() + (1.-tZ) * mGaP.me() - tZ * (1.-tZ) * 0.01854;
		return ( (mIn*mAl*tmeAlInP + mAl*mGa*tmeAlGaP + mGa*mIn*tmeInGaP)/(mIn*mAl + mAl*mGa + mGa*mIn) );		
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double AlGaInP::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tmhhLin = mAl * mAlP.mhh() + mGa * mGaP.mhh() + mIn * mInP.mhh();
	return ( tmhhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tmlhLin = mAl * mAlP.mlh() + mGa * mGaP.mlh() + mIn * mInP.mlh();
	return ( tmlhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tmsoLin = mAl * mAlP.mso() + mGa * mGaP.mso() + mIn * mInP.mso();
	return ( tmsoLin ); // lin = nlin1 = nlin2 
}
//------------------------------------------------------------------------------
double AlGaInP::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double AlGaInP::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP; punktem odniesienia jest Ev=0eV dla InSb";
	}
	double tEvLin = mAl * mAlP.Ev() + mGa * mGaP.Ev() + mIn * mInP.Ev();
	return ( tEvLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double AlGaInP::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tacLin = mAl * mAlP.ac() + mGa * mGaP.ac() + mIn * mInP.ac();
	return ( tacLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tavLin = mAl * mAlP.av() + mGa * mGaP.av() + mIn * mInP.av();
	return ( tavLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tbLin = mAl * mAlP.b() + mGa * mGaP.b() + mIn * mInP.b();
	return ( tbLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tdLin = mAl * mAlP.d() + mGa * mGaP.d() + mIn * mInP.d();
	return ( tdLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tc11Lin = mAl * mAlP.c11() + mGa * mGaP.c11() + mIn * mInP.c11();
	return ( tc11Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tc12Lin = mAl * mAlP.c12() + mGa * mGaP.c12() + mIn * mInP.c12();
	return ( tc12Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tc44Lin = mAl * mAlP.c44() + mGa * mGaP.c44() + mIn * mInP.c44();
	return ( tc44Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double AlGaInP::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double AlGaInP::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double AlGaInP::n(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	if (iDop == "Si")
	{
		double tn1 = Nc(iT) * exp ( -10e-3 / (cPhys::kB_eV*iT) ); // 10 meV -> J.R. Dong et al., Applied Physics Letters, 83, no.4, 2003.
		return ( (tn1/4.)*(sqrt(1.+8.*iN/tn1)-1.) ); // gD = 2
	}
	else if (iDop == "Zn")
	{
		double tp1 = Nv(iT) * exp ( -44e-3 / (cPhys::kB_eV*iT) ); // 44 meV -> J.R. Dong et al., Applied Physics Letters, 83, no.4, 2003.
		return ( (tp1/8.)*(sqrt(1.+16.*iN/tp1)-1.) ); // gA = 4
	}
	else		
		return iN; // TODO
}
//------------------------------------------------------------------------------
double AlGaInP::mob(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if (iDop == "Si")
	{
		double tAlGaInP_miRT = 4200e-4/(1+pow(n(300., iN, iDop)/1e22,0.97)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tAlGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "Se")
	{
		double tAlGaInP_miRT = 100e-4; // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tAlGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "Zn")
	{
		double tAlGaInP_miRT = 17e-4/(1+pow(n(300., iN, iDop)/4e24,3.09)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tAlGaInP_miRT * pow(300./iT,0.5) ); // (0.5 -> ref: praca magisterska Lukasza)		  
	}
	else if (iDop == "Be")
	{
		double tAlGaInP_miRT = 14e-4; // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tAlGaInP_miRT * pow(300./iT,0.5) ); // (0.5 -> ref: praca magisterska Lukasza)		  
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double AlGaInP::ec(double iT, double iN, std::string iDop, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cPhys::e * mob(iT, iN, iDop) * n(iT, iN, iDop) );       
}
//------------------------------------------------------------------------------
double AlGaInP::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	} 
	return ( 1./(mAl/mAlP.k(iT) + mGa/mGaP.k(iT) + mIn/mInP.k(iT) + mAl*mGa*0.32 + mAl*mIn*0.15 + mGa*mIn*0.72) );
}
//------------------------------------------------------------------------------
double AlGaInP::cp(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tcpLin = mAl * mAlP.cp(iT) + mGa * mGaP.cp(iT) + mIn * mInP.cp(iT);
	return ( tcpLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double AlGaInP::g(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla AlP, GaP i InP";
	}
	double tgLin = mAl * mAlP.g(iT) + mGa * mGaP.g(iT) + mIn * mInP.g(iT);
	return ( tgLin ); // lin = nlin1 = nlin2
}
//---------------------------------------------------------------------------
void AlGaInP::setInt(std::string iInt)
{
	mInt = iInt;
}
//---------------------------------------------------------------------------
AlGaInP::AlGaInP(double iAl,double iGa)
{
	mAl = iAl;
	mGa = iGa;
	mIn = 1.-iAl-iGa;
	mInt = "nlin1";
}
//---------------------------------------------------------------------------
AlGaInP::AlGaInP()
{  
}
//---------------------------------------------------------------------------
AlGaInP::~AlGaInP()
{
}

