#include "pch.h"

#include "GaInAsP.h"
//------------------------------------------------------------------------------
double GaInAsP::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	return ( mGa*mAs*mGaAs.a(iT) + mGa*mP*mGaP.a(iT) + mIn*mAs*mInAs.a(iT) + mIn*mP*mInP.a(iT) );
}
//------------------------------------------------------------------------------
double GaInAsP::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tEgLin = mGa*mAs*mGaAs.Eg(iT, iPoint) + mGa*mP*mGaP.Eg(iT, iPoint) + mIn*mAs*mInAs.Eg(iT, iPoint) + mIn*mP*mInP.Eg(iT, iPoint);
	if (mInt == "lin") // interpolacja liniowa
		return ( tEgLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
	{
		if (iPoint == 'G') return ( tEgLin - mGa*mIn*mAs*0.477 - mGa*mIn*mP*0.65 - mGa*mAs*mP*0.19 - mIn*mAs*mP*0.10 - mGa*mIn*mAs*mP*0.13 );
		else if (iPoint == 'X') return ( tEgLin - mGa*mIn*mAs*1.4 - mGa*mIn*mP*0.20 - mGa*mAs*mP*0.24 - mIn*mAs*mP*0.27 );
		else if (iPoint == 'L') return ( tEgLin - mGa*mIn*mAs*0.33 - mGa*mIn*mP*1.03 - mGa*mAs*mP*0.16 - mIn*mAs*mP*0.27 );
		else return 0.;
	}
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		if (iPoint == 'G')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 0.477;
			double tEgGaInP = mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 0.65;
			double tEgGaAsP = mAs * mGaAs.Eg(iT,iPoint) + mP * mGaP.Eg(iT,iPoint) - mAs * mP * 0.19;
			double tEgInAsP = mAs * mInAs.Eg(iT,iPoint) + mP * mInP.Eg(iT,iPoint) - mAs * mP * 0.10;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mP*tEgGaInP-mAs*mP*0.13) + (mAs*mP*(mGa*tEgGaAsP+mIn*tEgInAsP-mGa*mIn*0.13))) / (mGa*mIn+mAs*mP) );
		}
		else if (iPoint == 'X')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 1.4;
			double tEgGaInP = mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 0.20;
			double tEgGaAsP = mAs * mGaAs.Eg(iT,iPoint) + mP * mGaP.Eg(iT,iPoint) - mAs * mP * 0.24;
			double tEgInAsP = mAs * mInAs.Eg(iT,iPoint) + mP * mInP.Eg(iT,iPoint) - mAs * mP * 0.27;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mP*tEgGaInP) + (mAs*mP*(mGa*tEgGaAsP+mIn*tEgInAsP))) / (mGa*mIn+mAs*mP) );
		}		
		else if (iPoint == 'L')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 0.33;
			double tEgGaInP = mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 1.03;
			double tEgGaAsP = mAs * mGaAs.Eg(iT,iPoint) + mP * mGaP.Eg(iT,iPoint) - mAs * mP * 0.16;
			double tEgInAsP = mAs * mInAs.Eg(iT,iPoint) + mP * mInP.Eg(iT,iPoint) - mAs * mP * 0.27;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mP*tEgGaInP) + (mAs*mP*(mGa*tEgGaAsP+mIn*tEgInAsP))) / (mGa*mIn+mAs*mP) );
		}		
		else return 0.;
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsP::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double GaInAsP::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tDsoLin = mGa*mAs*mGaAs.Dso() + mGa*mP*mGaP.Dso() + mIn*mAs*mInAs.Dso() + mIn*mP*mInP.Dso();
	if (mInt == "lin") // interpolacja liniowa
		return ( tDsoLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tDsoLin - mGa*mIn*mAs*0.15 /*- mGa*mIn*mP*0.*/ /*- mGa*mAs*mP*0.*/ - mIn*mAs*mP*0.16 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tDsoGaInAs = mGa * mGaAs.Dso() + mIn * mInAs.Dso() - mGa * mIn * 0.15;
		double tDsoGaInP = mGa * mGaP.Dso() + mIn * mInP.Dso() /*- mGa * mIn * 0.*/;
		double tDsoGaAsP = mAs * mGaAs.Dso() + mP * mGaP.Dso() /*- mAs * mP * 0.*/;
		double tDsoInAsP = mAs * mInAs.Dso() + mP * mInP.Dso() - mAs * mP * 0.16;
		return ( (mGa*mIn*(mAs*tDsoGaInAs+mP*tDsoGaInP) + (mAs*mP*(mGa*tDsoGaAsP+mIn*tDsoInAsP))) / (mGa*mIn+mAs*mP) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsP::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tmeLin = mGa*mAs*mGaAs.me() + mGa*mP*mGaP.me() + mIn*mAs*mInAs.me() + mIn*mP*mInP.me();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmeLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmeLin - mGa*mIn*mAs*0.008 - mGa*mIn*mP*0.01854 /*- mGa*mAs*mP*0.*/ /*- mIn*mAs*mP*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tmeGaInAs = mGa * mGaAs.me() + mIn * mInAs.me() - mGa * mIn * 0.008;
		double tmeGaInP = mGa * mGaP.me() + mIn * mInP.me() - mGa * mIn * 0.01854;
		double tmeGaAsP = mAs * mGaAs.me() + mP * mGaP.me() /*- mAs * mP * 0.*/;
		double tmeInAsP = mAs * mInAs.me() + mP * mInP.me() /*- mAs * mP * 0.*/;
		return ( (mGa*mIn*(mAs*tmeGaInAs+mP*tmeGaInP) + (mAs*mP*(mGa*tmeGaAsP+mIn*tmeInAsP))) / (mGa*mIn+mAs*mP) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsP::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tmhhLin = mGa*mAs*mGaAs.mhh() + mGa*mP*mGaP.mhh() + mIn*mAs*mInAs.mhh() + mIn*mP*mInP.mhh();
	return ( tmhhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tmlhLin = mGa*mAs*mGaAs.mlh() + mGa*mP*mGaP.mlh() + mIn*mAs*mInAs.mlh() + mIn*mP*mInP.mlh();
	return ( tmlhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tmsoLin = mGa*mAs*mGaAs.mso() + mGa*mP*mGaP.mso() + mIn*mAs*mInAs.mso() + mIn*mP*mInP.mso();
	return ( tmsoLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double GaInAsP::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP; punktem odniesienia jest Ev=0eV dla InSb";
	}
	double tEvLin = mGa*mAs*mGaAs.Ev() + mGa*mP*mGaP.Ev() + mIn*mAs*mInAs.Ev() + mIn*mP*mInP.Ev();
	if (mInt == "lin") // interpolacja liniowa
		return ( tEvLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tEvLin - mGa*mIn*mAs*(-0.38) /*- mGa*mIn*mP*0.*/ /*- mGa*mAs*mP*0.*/ /*- mIn*mAs*mP*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tEvGaInAs = mGa * mGaAs.Ev() + mIn * mInAs.Ev() - mGa * mIn * (-0.38);
		double tEvGaInP = mGa * mGaP.Ev() + mIn * mInP.Ev() /*- mGa * mIn * 0.*/;
		double tEvGaAsP = mAs * mGaAs.Ev() + mP * mGaP.Ev() /*- mAs * mP * 0.*/;
		double tEvInAsP = mAs * mInAs.Ev() + mP * mInP.Ev() /*- mAs * mP * 0.*/;
		return ( (mGa*mIn*(mAs*tEvGaInAs+mP*tEvGaInP) + (mAs*mP*(mGa*tEvGaAsP+mIn*tEvInAsP))) / (mGa*mIn+mAs*mP) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsP::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double GaInAsP::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tacLin = mGa*mAs*mGaAs.ac() + mGa*mP*mGaP.ac() + mIn*mAs*mInAs.ac() + mIn*mP*mInP.ac();
	if (mInt == "lin") // interpolacja liniowa
		return ( tacLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tacLin - mGa*mIn*mAs*2.61 /*- mGa*mIn*mP*0.*/ /*- mGa*mAs*mP*0.*/ /*- mIn*mAs*mP*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tacGaInAs = mGa * mGaAs.ac() + mIn * mInAs.ac() - mGa * mIn * 2.61;
		double tacGaInP = mGa * mGaP.ac() + mIn * mInP.ac() /*- mGa * mIn * 0.*/;
		double tacGaAsP = mAs * mGaAs.ac() + mP * mGaP.ac() /*- mAs * mP * 0.*/;
		double tacInAsP = mAs * mInAs.ac() + mP * mInP.ac() /*- mAs * mP * 0.*/;
		return ( (mGa*mIn*(mAs*tacGaInAs+mP*tacGaInP) + (mAs*mP*(mGa*tacGaAsP+mIn*tacInAsP))) / (mGa*mIn+mAs*mP) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsP::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tavLin = mGa*mAs*mGaAs.av() + mGa*mP*mGaP.av() + mIn*mAs*mInAs.av() + mIn*mP*mInP.av();
	return ( tavLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tbLin = mGa*mAs*mGaAs.b() + mGa*mP*mGaP.b() + mIn*mAs*mInAs.b() + mIn*mP*mInP.b();
	return ( tbLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tdLin = mGa*mAs*mGaAs.d() + mGa*mP*mGaP.d() + mIn*mAs*mInAs.d() + mIn*mP*mInP.d();
	return ( tdLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tc11Lin = mGa*mAs*mGaAs.c11() + mGa*mP*mGaP.c11() + mIn*mAs*mInAs.c11() + mIn*mP*mInP.c11();
	return ( tc11Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tc12Lin = mGa*mAs*mGaAs.c12() + mGa*mP*mGaP.c12() + mIn*mAs*mInAs.c12() + mIn*mP*mInP.c12();
	return ( tc12Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tc44Lin = mGa*mAs*mGaAs.c44() + mGa*mP*mGaP.c44() + mIn*mAs*mInAs.c44() + mIn*mP*mInP.c44();
	return ( tc44Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double GaInAsP::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double GaInAsP::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double GaInAsP::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	} 
	return ( 1./(mGa*mAs/mGaAs.k(iT) + mGa*mP/mGaP.k(iT) + mIn*mAs/mInAs.k(iT) + mIn*mP/mInP.k(iT) + mGa*mIn*0.72 + mAs*mP*0.91) ); // 0.91 -> tymczasowo tak jak dla As-Sb
}
//------------------------------------------------------------------------------
double GaInAsP::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tcpLin = mGa*mAs*mGaAs.cp(iT) + mGa*mP*mGaP.cp(iT) + mIn*mAs*mInAs.cp(iT) + mIn*mP*mInP.cp(iT);
	return ( tcpLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsP::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaP, InAs i InP";
	}
	double tgLin = mGa*mAs*mGaAs.g(iT) + mGa*mP*mGaP.g(iT) + mIn*mAs*mInAs.g(iT) + mIn*mP*mInP.g(iT);
	return ( tgLin ); // lin = nlin1 = nlin2
}
//---------------------------------------------------------------------------
double GaInAsP::nR(double iLam, double iT, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. LINNIK, A.CHRISTOU";
		ipRef->title="OPTICAL PROPERTIES OF AlGaAsSb, AlGalnP, AlGaInAs, AND GaInAsP...";
		ipRef->journal="Mat. Res. Soc. Symp. Proc. Vol. 579 pp.201, 2000";
		ipRef->comment="zaleznosc oblicza nR dla fal 1100-2000 nm";
	}
	if ( (iLam>=1100.0) && (iLam<=2000.0) )
	{
		double Pi = cMath::pi;
		double h  = 6.62607004e-34;
		double c  = 299792458;
		double q  = 1.6021766208e-19;

		double E_photon = h*c/(q*iLam*1e-9);	//[eV]

		//Ternary alloy material parameters: bowing parameters 
		double C_AlGaAs_G = 0.37;
		double C_AlInAs_G = 0.52;
		double C_GaInAs_G = 1.4;
		double C_AlGaP_G  = 0.0;
		double C_AlInP_G  = 0.0;
		double C_GaInP_G  = 0.18;
		double C_AlGaSb_G = 0.47;
		double C_GaAsP_G  = 0.21;
		double C_AlSbAs_G = 0.0;
		double C_GaSbAs_G = 1.2;
		double C_InAsP_G  = 0.27;

		double C_AlGaAs_X = 0.245;
		double C_AlInAs_X = 0.0;
		double C_GaInAs_X = 1.4;
		double C_AlGaP_X  = 0.0;
		double C_AlInP_X  = 0.0;
		double C_GaInP_X  = 0.15;
		double C_AlGaSb_X = 0.0;
		double C_GaAsP_X  = 0.21;
		double C_AlSbAs_X = 0.0;
		double C_GaSbAs_X = 1.09;
		double C_InAsP_X  = 0.27;

		double C_AlGaAs_L = 0.055;
		double C_AlInAs_L = 0.0;
		double C_GaInAs_L = 0.72;
		double C_AlGaP_L  = 0.0;
		double C_AlInP_L  = 0.0;
		double C_GaInP_L  = 0.43;
		double C_AlGaSb_L = 0.55;
		double C_GaAsP_L  = 0.42;
		double C_AlSbAs_L = 0.0;
		double C_GaSbAs_L = 1.09;
		double C_InAsP_L  = 0.27;

		//Binary alloy material parameters: direct and indirect energy bandgaps E_G,E_X, E_L [eV], splitting energies DEL_SO [eV] A, and strength parameters A and B
		/*
		AlAs 2.95 2.16 2.36 0.28 25.30		-0.8
		AlP  3.63 2.45 4.00 0.00 18.31		 0.59
		AlSb 2.30 1.61 2.21 0.72 59.68		-9.53
		GaAs 1.43 1.91 1.73 0.34 9.29(5.3*)  7.86(10.4*)
		GaP  2.74 2.27 3.00 0.13 22.25		 0.9
		GaSb 0.72 1.05 0.76 0.74 4.05		 12.66
		InAs 0.36 1.37 1.07 0.40 4.36(6.14*) 11.65(9.15*)
		InP  1.35 2.04 1.87 0.11 8.40		 6.60
		*/
		//* Values of A and B used in calculations for AlGaInAs system.

		double E_AlAs_G = 2.95;
		double E_AlP_G  = 3.63;
		double E_AlSb_G = 2.30;
		double E_GaAs_G = 1.43;
		double E_GaP_G  = 2.74;
		double E_GaSb_G = 0.72;
		double E_InAs_G = 0.36;
		double E_InP_G  = 1.35;

		double E_AlAs_X = 2.16;
		double E_AlP_X  = 2.45;
		double E_AlSb_X = 1.61;
		double E_GaAs_X = 1.91;
		double E_GaP_X  = 2.27;
		double E_GaSb_X = 1.05;
		double E_InAs_X = 1.37;
		double E_InP_X  = 2.04;

		double E_AlAs_L = 2.36;
		double E_AlP_L  = 4.00;
		double E_AlSb_L = 2.21;
		double E_GaAs_L = 1.73;
		double E_GaP_L  = 3.00;
		double E_GaSb_L = 0.76;
		double E_InAs_L = 1.07;
		double E_InP_L  = 1.87;

		double D_SO_AlAs = 0.28;
		double D_SO_AlP  = 0.00;
		double D_SO_AlSb = 0.72;
		double D_SO_GaAs = 0.34;
		double D_SO_GaP  = 0.13;
		double D_SO_GaSb = 0.74;
		double D_SO_InAs = 0.40;
		double D_SO_InP  = 0.11;

		double A_AlAs = 25.30;
		double A_AlP  = 18.31;
		double A_AlSb = 59.68;
		double A_GaAs = 9.29;
		double A_GaP  = 22.25;
		double A_GaSb = 4.05;
		double A_InAs = 4.36;
		double A_InP  = 8.40;

		double B_AlAs = -0.80;
		double B_AlP  = 0.59;
		double B_AlSb = -9.53;
		double B_GaAs = 7.86;
		double B_GaP  = 0.90;
		double B_GaSb = 12.66;
		double B_InAs = 11.65;
		double B_InP  = 6.60;

		//dla Ga_xIn_1-xAs_yP_1-y -		A(x,y) = x*y*A_AC + x*(1-y)*A_AD +(1-x)*y*A_BC + (1-x)*(1-y)*A_BD
		double D_SO_xy	= mGa*mAs*D_SO_GaAs + mGa*(1-mAs)*D_SO_GaP +(1-mGa)*mAs*D_SO_InAs + (1-mGa)*(1-mAs)*D_SO_InP;
		double A_xy		= mGa*mAs*A_GaAs + mGa*(1-mAs)*A_GaP +(1-mGa)*mAs*A_InAs + (1-mGa)*(1-mAs)*A_InP;
		double B_xy		= mGa*mAs*B_GaAs + mGa*(1-mAs)*B_GaP +(1-mGa)*mAs*B_InAs + (1-mGa)*(1-mAs)*B_InP;

		double E_xy_G, E_xy_X, E_xy_L;

		if((mGa == 0.0) && (mAs == 0.0))		//dla InP
		{
			E_xy_G	= E_InP_G;
			E_xy_X	= E_InP_X;
			E_xy_L	= E_InP_L;
		}
		else if((mGa == 1.0) && (mAs == 1.0))		//dla GaAs
		{
			E_xy_G	= E_GaAs_G;
			E_xy_X	= E_GaAs_X;
			E_xy_L	= E_GaAs_L;
		}
		else if((mGa == 1.0) && (mAs == 0.0))		//dla GaP
		{
			E_xy_G	= E_GaP_G;
			E_xy_X	= E_GaP_X;
			E_xy_L	= E_GaP_L;
		}
		else if((mGa == 0.0) && (mAs == 1.0))		//dla InAs
		{
			E_xy_G	= E_InAs_G;
			E_xy_X	= E_InAs_X;
			E_xy_L	= E_InAs_L;
		}
		else if((mGa == 0.0) && (mAs != 0.0))	//dla InAsP
		{
			E_xy_G	= mAs*E_InAs_G + (1-mAs)*E_InP_G  - mAs*(1-mAs)*C_InAsP_G;
			E_xy_X	= mAs*E_InAs_X + (1-mAs)*E_InP_X  - mAs*(1-mAs)*C_InAsP_X;
			E_xy_L	= mAs*E_InAs_L + (1-mAs)*E_InP_L  - mAs*(1-mAs)*C_InAsP_L;
		}
		else if((mGa != 0.0) && (mAs == 0.0))	//dla GaInP
		{
			E_xy_G	= mGa*E_GaP_G  + (1-mGa)*E_InP_G  - mGa*(1-mGa)*C_GaInP_G;
			E_xy_X	= mGa*E_GaP_X  + (1-mGa)*E_InP_X  - mGa*(1-mGa)*C_GaInP_X;
			E_xy_L	= mGa*E_GaP_L  + (1-mGa)*E_InP_L  - mGa*(1-mGa)*C_GaInP_L;
		}
		else if((mGa == 1.0) && (mAs != 0.0))	//dla GaAsP
		{
			E_xy_G	= mAs*E_GaAs_G  + (1-mAs)*E_GaP_G  - mAs*(1-mAs)*C_GaAsP_G;
			E_xy_X	= mAs*E_GaAs_X  + (1-mAs)*E_GaP_X  - mAs*(1-mAs)*C_GaAsP_X;
			E_xy_L	= mAs*E_GaAs_L  + (1-mAs)*E_GaP_L  - mAs*(1-mAs)*C_GaAsP_L;
		}
		else if((mGa != 0.0) && (mAs == 1.0))	//dla GaInAs
		{
			E_xy_G	= mGa*E_GaAs_G  + (1-mGa)*E_InAs_G  - mGa*(1-mGa)*C_GaInAs_G;
			E_xy_X	= mGa*E_GaAs_X  + (1-mGa)*E_InAs_X  - mGa*(1-mGa)*C_GaInAs_X;
			E_xy_L	= mGa*E_GaAs_L  + (1-mGa)*E_InAs_L  - mGa*(1-mGa)*C_GaInAs_L;
		}
		else
		{
			//dla Ga_xIn_1-xAs_yP_1-y -		Eg(x,y) = (x*(1-x)*((1-y)*E_ABD_x + y*E_ABC_x) + y*(1-y)*((1-x)*E_BCD_y + x*E_ACD_y)) / (x*(1-x)+y*(1-y))
			//								E_ABC_x = x*E_AB + (1-x)*E_AC - x*(1-x)*C_ABC
			//								itd.
			double E_GaInP_G	= mGa*E_GaP_G  + (1-mGa)*E_InP_G  - mGa*(1-mGa)*C_GaInP_G;
			double E_GaInAs_G	= mGa*E_GaAs_G + (1-mGa)*E_InAs_G - mGa*(1-mGa)*C_GaInAs_G;
			double E_InAsP_G	= mAs*E_InAs_G + (1-mAs)*E_InP_G  - mAs*(1-mAs)*C_InAsP_G;
			double E_GaAsP_G	= mAs*E_GaAs_G + (1-mAs)*E_GaP_G  - mAs*(1-mAs)*C_GaAsP_G;

			E_xy_G	= (mGa*(1-mGa)*((1-mAs)*E_GaInP_G + mAs*E_GaInAs_G) + mAs*(1-mAs)*((1-mGa)*E_InAsP_G + mGa*E_GaAsP_G)) / (mGa*(1-mGa)+mAs*(1-mAs));


			double E_GaInP_X	= mGa*E_GaP_X  + (1-mGa)*E_InP_X  - mGa*(1-mGa)*C_GaInP_X;
			double E_GaInAs_X	= mGa*E_GaAs_X + (1-mGa)*E_InAs_X - mGa*(1-mGa)*C_GaInAs_X;
			double E_InAsP_X	= mAs*E_InAs_X + (1-mAs)*E_InP_X  - mAs*(1-mAs)*C_InAsP_X;
			double E_GaAsP_X	= mAs*E_GaAs_X + (1-mAs)*E_GaP_X  - mAs*(1-mAs)*C_GaAsP_X;

			E_xy_X	= (mGa*(1-mGa)*((1-mAs)*E_GaInP_X + mAs*E_GaInAs_X) + mAs*(1-mAs)*((1-mGa)*E_InAsP_X + mGa*E_GaAsP_X)) / (mGa*(1-mGa)+mAs*(1-mAs));


			double E_GaInP_L	= mGa*E_GaP_L  + (1-mGa)*E_InP_L  - mGa*(1-mGa)*C_GaInP_L;
			double E_GaInAs_L	= mGa*E_GaAs_L + (1-mGa)*E_InAs_L - mGa*(1-mGa)*C_GaInAs_L;
			double E_InAsP_L	= mAs*E_InAs_L + (1-mAs)*E_InP_L  - mAs*(1-mAs)*C_InAsP_L;
			double E_GaAsP_L	= mAs*E_GaAs_L + (1-mAs)*E_GaP_L  - mAs*(1-mAs)*C_GaAsP_L;

			E_xy_L	= (mGa*(1-mGa)*((1-mAs)*E_GaInP_L + mAs*E_GaInAs_L) + mAs*(1-mAs)*((1-mGa)*E_InAsP_L + mGa*E_GaAsP_L)) / (mGa*(1-mGa)+mAs*(1-mAs));
		}

		//wybieramy najmniejsze E_xy
		double E_xy		= E_xy_G;
		if(E_xy_X < E_xy) E_xy = E_xy_X;
		if(E_xy_L < E_xy) E_xy = E_xy_L;

		double Ksi0  = E_photon/E_xy;
		double KsiSO = E_photon/(E_xy+D_SO_xy);

		double H0 = 1.0;
		if((1-Ksi0)<0)	H0 = 0.0;
		else			H0 = sqrt(1 - Ksi0);

		double HSO = 1.0;
		if((1-KsiSO)<0) HSO = 0.0;
		else			HSO = sqrt(1 - KsiSO);

		double F_Ksi0 = (2 - sqrt(1 + Ksi0) - H0)/(Ksi0*Ksi0);
		double F_KsiSO = (2 - sqrt(1 + KsiSO) - HSO)/(KsiSO*KsiSO);

		double nR = sqrt(A_xy*(F_Ksi0 + 0.5*pow(E_xy/(E_xy+D_SO_xy), 1.5)*F_KsiSO) + B_xy);

		//zaleznosc temperaturowa na bazie pracy:
		//Abi Waqas, Daniele Melati, Zarlish Mushtaq, B.S. Chowdhry, Andrea Melloni;
		//An Improved Model to Predict the Temperature Dependence of Refractive Index of InP-based Compounds
		//Wireless Pers Commun. 2016; DOI 10.1007/s11277-016-3913-5
		double dn_dT = 1e-4;	//1/K

		if(iLam < 1250) dn_dT *= 2.3;
		else if(iLam < 1350) dn_dT *= 2.1;
		else if(iLam < 1450) dn_dT *= 1.97;
		else dn_dT *= 1.9;

		return nR + dn_dT*(iT-300);
	}
	else
		return 0.;
}
//---------------------------------------------------------------------------
void GaInAsP::setInt(std::string iInt)
{
	mInt = iInt;
}
//---------------------------------------------------------------------------
GaInAsP::GaInAsP(double iGa,double iAs)
{
	mGa = iGa;
	mIn = 1.-iGa;
	mAs = iAs;
	mP = 1.-mAs;
	mInt = "nlin1";
}
//---------------------------------------------------------------------------
GaInAsP::GaInAsP()
{  
}
//---------------------------------------------------------------------------
GaInAsP::~GaInAsP()
{
}
