#include "pch.h"

#include "GaInAsSb.h"
//------------------------------------------------------------------------------
double GaInAsSb::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	return ( mGa*mAs*mGaAs.a(iT) + mGa*mSb*mGaSb.a(iT) + mIn*mAs*mInAs.a(iT) + mIn*mSb*mInSb.a(iT) );
}
//------------------------------------------------------------------------------
double GaInAsSb::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tEgLin = mGa*mAs*mGaAs.Eg(iT, iPoint) + mGa*mSb*mGaSb.Eg(iT, iPoint) + mIn*mAs*mInAs.Eg(iT, iPoint) + mIn*mSb*mInSb.Eg(iT, iPoint);
	if (mInt == "lin") // interpolacja liniowa
		return ( tEgLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
	{
		if (iPoint == 'G') return ( tEgLin - mGa*mIn*mAs*0.477 - mGa*mIn*mSb*0.415 - mGa*mAs*mSb*1.43 - mIn*mAs*mSb*0.67 - mGa*mIn*mAs*mSb*0.75 );
		else if (iPoint == 'X') return ( tEgLin - mGa*mIn*mAs*1.4 - mGa*mIn*mSb*0.33 - mGa*mAs*mSb*1.2 - mIn*mAs*mSb*0.6 );
		else if (iPoint == 'L') return ( tEgLin - mGa*mIn*mAs*0.33 - mGa*mIn*mSb*0.4 - mGa*mAs*mSb*1.2 - mIn*mAs*mSb*0.6 );
		else return 0.;
	}
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		if (iPoint == 'G')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 0.477;
			double tEgGaInSb = mGa * mGaSb.Eg(iT,iPoint) + mIn * mInSb.Eg(iT,iPoint) - mGa * mIn * 0.415;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.43;
			double tEgInAsSb = mAs * mInAs.Eg(iT,iPoint) + mSb * mInSb.Eg(iT,iPoint) - mAs * mSb * 0.67;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mSb*tEgGaInSb-mAs*mSb*0.75) + (mAs*mSb*(mGa*tEgGaAsSb+mIn*tEgInAsSb-mGa*mIn*0.75))) / (mGa*mIn+mAs*mSb) );
		}
		else if (iPoint == 'X')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 1.4;
			double tEgGaInSb = mGa * mGaSb.Eg(iT,iPoint) + mIn * mInSb.Eg(iT,iPoint) - mGa * mIn * 0.33;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.2;
			double tEgInAsSb = mAs * mInAs.Eg(iT,iPoint) + mSb * mInSb.Eg(iT,iPoint) - mAs * mSb * 0.6;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mSb*tEgGaInSb) + (mAs*mSb*(mGa*tEgGaAsSb+mIn*tEgInAsSb))) / (mGa*mIn+mAs*mSb) );
		}		
		else if (iPoint == 'L')
		{
			double tEgGaInAs = mGa * mGaAs.Eg(iT,iPoint) + mIn * mInAs.Eg(iT,iPoint) - mGa * mIn * 0.33;
			double tEgGaInSb = mGa * mGaSb.Eg(iT,iPoint) + mIn * mInSb.Eg(iT,iPoint) - mGa * mIn * 0.4;
			double tEgGaAsSb = mAs * mGaAs.Eg(iT,iPoint) + mSb * mGaSb.Eg(iT,iPoint) - mAs * mSb * 1.2;
			double tEgInAsSb = mAs * mInAs.Eg(iT,iPoint) + mSb * mInSb.Eg(iT,iPoint) - mAs * mSb * 0.6;
			return ( (mGa*mIn*(mAs*tEgGaInAs+mSb*tEgGaInSb) + (mAs*mSb*(mGa*tEgGaAsSb+mIn*tEgInAsSb))) / (mGa*mIn+mAs*mSb) );
		}		
		else return 0.;
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tEgG = Eg(iT, 'G');
	double tEgX = Eg(iT, 'X');
	double tEgL = Eg(iT, 'L');
	if ((tEgG <= tEgX) && (tEgG <= tEgL)) return tEgG;
	else if (tEgX <= tEgL) return tEgX;
	else return tEgL;
}
//------------------------------------------------------------------------------
double GaInAsSb::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tDsoLin = mGa*mAs*mGaAs.Dso() + mGa*mSb*mGaSb.Dso() + mIn*mAs*mInAs.Dso() + mIn*mSb*mInSb.Dso();
	if (mInt == "lin") // interpolacja liniowa
		return ( tDsoLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tDsoLin - mGa*mIn*mAs*0.15 - mGa*mIn*mSb*0.1 - mGa*mAs*mSb*0.6 - mIn*mAs*mSb*1.2 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tDsoGaInAs = mGa * mGaAs.Dso() + mIn * mInAs.Dso() - mGa * mIn * 0.15;
		double tDsoGaInSb = mGa * mGaSb.Dso() + mIn * mInSb.Dso() - mGa * mIn * 0.1;
		double tDsoGaAsSb = mAs * mGaAs.Dso() + mSb * mGaSb.Dso() - mAs * mSb * 0.6;
		double tDsoInAsSb = mAs * mInAs.Dso() + mSb * mInSb.Dso() - mAs * mSb * 1.2;
		return ( (mGa*mIn*(mAs*tDsoGaInAs+mSb*tDsoGaInSb) + (mAs*mSb*(mGa*tDsoGaAsSb+mIn*tDsoInAsSb))) / (mGa*mIn+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tmeLin = mGa*mAs*mGaAs.me() + mGa*mSb*mGaSb.me() + mIn*mAs*mInAs.me() + mIn*mSb*mInSb.me();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmeLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmeLin - mGa*mIn*mAs*0.008 - mGa*mIn*mSb*0.010 - mGa*mAs*mSb*0.014 - mIn*mAs*mSb*0.027 );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tmeGaInAs = mGa * mGaAs.me() + mIn * mInAs.me() - mGa * mIn * 0.008;
		double tmeGaInSb = mGa * mGaSb.me() + mIn * mInSb.me() - mGa * mIn * 0.010;
		double tmeGaAsSb = mAs * mGaAs.me() + mSb * mGaSb.me() - mAs * mSb * 0.014;
		double tmeInAsSb = mAs * mInAs.me() + mSb * mInSb.me() - mAs * mSb * 0.027;
		return ( (mGa*mIn*(mAs*tmeGaInAs+mSb*tmeGaInSb) + (mAs*mSb*(mGa*tmeGaAsSb+mIn*tmeInAsSb))) / (mGa*mIn+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tmhhLin = mGa*mAs*mGaAs.mhh() + mGa*mSb*mGaSb.mhh() + mIn*mAs*mInAs.mhh() + mIn*mSb*mInSb.mhh();
	return ( tmhhLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tmlhLin = mGa*mAs*mGaAs.mlh() + mGa*mSb*mGaSb.mlh() + mIn*mAs*mInAs.mlh() + mIn*mSb*mInSb.mlh();
	if (mInt == "lin") // interpolacja liniowa
		return ( tmlhLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tmlhLin /*- mGa*mIn*mAs*0.*/ - mGa*mIn*mSb*0.015 /*- mGa*mAs*mSb*0.*/ /*- mIn*mAs*mSb*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tmlhGaInAs = mGa * mGaAs.mlh() + mIn * mInAs.mlh() /*- mGa * mIn * 0.*/;
		double tmlhGaInSb = mGa * mGaSb.mlh() + mIn * mInSb.mlh() - mGa * mIn * 0.015;
		double tmlhGaAsSb = mAs * mGaAs.mlh() + mSb * mGaSb.mlh() /*- mAs * mSb * 0.*/;
		double tmlhInAsSb = mAs * mInAs.mlh() + mSb * mInSb.mlh() /*- mAs * mSb * 0.*/;
		return ( (mGa*mIn*(mAs*tmlhGaInAs+mSb*tmlhGaInSb) + (mAs*mSb*(mGa*tmlhGaAsSb+mIn*tmlhInAsSb))) / (mGa*mIn+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tmsoLin = mGa*mAs*mGaAs.mso() + mGa*mSb*mGaSb.mso() + mIn*mAs*mInAs.mso() + mIn*mSb*mInSb.mso();
	return ( tmsoLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double GaInAsSb::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb; punktem odniesienia jest Ev=0eV dla InSb";
	}
	double tEvLin = mGa*mAs*mGaAs.Ev() + mGa*mSb*mGaSb.Ev() + mIn*mAs*mInAs.Ev() + mIn*mSb*mInSb.Ev() ;
	if (mInt == "lin") // interpolacja liniowa
		return ( tEvLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tEvLin - mGa*mIn*mAs*(-0.38) /*- mGa*mIn*mSb*0.*/ - mGa*mAs*mSb*(-1.06) /*- mIn*mAs*mSb*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tEvGaInAs = mGa * mGaAs.Ev() + mIn * mInAs.Ev() - mGa * mIn * (-0.38);
		double tEvGaInSb = mGa * mGaSb.Ev() + mIn * mInSb.Ev() /*- mGa * mIn * 0.*/;
		double tEvGaAsSb = mAs * mGaAs.Ev() + mSb * mGaSb.Ev() - mAs * mSb * (-1.06);
		double tEvInAsSb = mAs * mInAs.Ev() + mSb * mInSb.Ev() /*- mAs * mSb * 0.*/;
		return ( (mGa*mIn*(mAs*tEvGaInAs+mSb*tEvGaInSb) + (mAs*mSb*(mGa*tEvGaAsSb+mIn*tEvInAsSb))) / (mGa*mIn+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double GaInAsSb::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tacLin = mGa*mAs*mGaAs.ac() + mGa*mSb*mGaSb.ac() + mIn*mAs*mInAs.ac() + mIn*mSb*mInSb.ac();
	if (mInt == "lin") // interpolacja liniowa
		return ( tacLin );
	else if (mInt == "nlin1") // interpolacja nieliniowa prosta 
		return ( tacLin - mGa*mIn*mAs*2.61 /*- mGa*mIn*mSb*0.*/ /*- mGa*mAs*mSb*0.*/ /*- mIn*mAs*mSb*0.*/ );
	else if (mInt == "nlin2") // interpolacja nieliniowa zlozona 
	{
		double tacGaInAs = mGa * mGaAs.ac() + mIn * mInAs.ac() - mGa * mIn * 2.61;
		double tacGaInSb = mGa * mGaSb.ac() + mIn * mInSb.ac() /*- mGa * mIn * 0.*/;
		double tacGaAsSb = mAs * mGaAs.ac() + mSb * mGaSb.ac() /*- mAs * mSb * 0.*/;
		double tacInAsSb = mAs * mInAs.ac() + mSb * mInSb.ac() /*- mAs * mSb * 0.*/;
		return ( (mGa*mIn*(mAs*tacGaInAs+mSb*tacGaInSb) + (mAs*mSb*(mGa*tacGaAsSb+mIn*tacInAsSb))) / (mGa*mIn+mAs*mSb) );
	}
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInAsSb::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tavLin = mGa*mAs*mGaAs.av() + mGa*mSb*mGaSb.av() + mIn*mAs*mInAs.av() + mIn*mSb*mInSb.av();
	return ( tavLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tbLin = mGa*mAs*mGaAs.b() + mGa*mSb*mGaSb.b() + mIn*mAs*mInAs.b() + mIn*mSb*mInSb.b();
	return ( tbLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tdLin = mGa*mAs*mGaAs.d() + mGa*mSb*mGaSb.d() + mIn*mAs*mInAs.d() + mIn*mSb*mInSb.d();
	return ( tdLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tc11Lin = mGa*mAs*mGaAs.c11() + mGa*mSb*mGaSb.c11() + mIn*mAs*mInAs.c11() + mIn*mSb*mInSb.c11();
	return ( tc11Lin ); // lin = nlin1 = nlin2 
}
//------------------------------------------------------------------------------
double GaInAsSb::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tc12Lin = mGa*mAs*mGaAs.c12() + mGa*mSb*mGaSb.c12() + mIn*mAs*mInAs.c12() + mIn*mSb*mInSb.c12();
	return ( tc12Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tc44Lin = mGa*mAs*mGaAs.c44() + mGa*mSb*mGaSb.c44() + mIn*mAs*mInAs.c44() + mIn*mSb*mInSb.c44();
	return ( tc44Lin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double GaInAsSb::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double GaInAsSb::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double GaInAsSb::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	} 
	return ( 1./(mGa*mAs/mGaAs.k(iT) + mGa*mSb/mGaSb.k(iT) + mIn*mAs/mInAs.k(iT) + mIn*mSb/mInSb.k(iT) + mGa*mIn*0.72 + mAs*mSb*0.91) );
}
//------------------------------------------------------------------------------
double GaInAsSb::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tcpLin = mGa*mAs*mGaAs.cp(iT) + mGa*mSb*mGaSb.cp(iT) + mIn*mAs*mInAs.cp(iT) + mIn*mSb*mInSb.cp(iT);
	return ( tcpLin ); // lin = nlin1 = nlin2
}
//------------------------------------------------------------------------------
double GaInAsSb::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaAs, GaSb, InAs i InSb";
	}
	double tgLin = mGa*mAs*mGaAs.g(iT) + mGa*mSb*mGaSb.g(iT) + mIn*mAs*mInAs.g(iT) + mIn*mSb*mInSb.g(iT);
	return ( tgLin ); // lin = nlin1 = nlin2
}
//---------------------------------------------------------------------------
double GaInAsSb::nR(double iLam, double iT, reference *ipRef) 
{
	//brak danych o InSb, InSbAs i GaInSb
	//*
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="M. LINNIK, A.CHRISTOU";
		ipRef->title="OPTICAL PROPERTIES OF AlGaAsSb, AlGalnP, AlGaInAs, AND GaInAsP...";
		ipRef->journal="Mat. Res. Soc. Symp. Proc. Vol. 579 pp.201, 2000";
		ipRef->comment="zaleznosc oblicza nR dla fal ok. 500-4000 nm";
	}
	if ( (iLam>=500.0) && (iLam<=4000.0) )
	{
		double Pi = cMath::pi;
		double h  = 6.62607004e-34;
		double c  = 299792458;
		double q  = 1.6021766208e-19;

		double E_photon = h*c/(q*iLam*1e-9);	//[eV]

		//Ternary alloy material parameters: bowing parameters 
		double C_AlGaAs_G = 0.37;
		double C_AlInAs_G = 0.52;
		double C_GaInAs_G = 1.4;
		double C_AlGaP_G  = 0.0;
		double C_AlInP_G  = 0.0;
		double C_GaInP_G  = 0.18;
		double C_AlGaSb_G = 0.47;
		double C_GaAsP_G  = 0.21;
		double C_AlSbAs_G = 0.0;
		double C_GaSbAs_G = 1.2;
		double C_InAsP_G  = 0.27;

		double C_AlGaAs_X = 0.245;
		double C_AlInAs_X = 0.0;
		double C_GaInAs_X = 1.4;
		double C_AlGaP_X  = 0.0;
		double C_AlInP_X  = 0.0;
		double C_GaInP_X  = 0.15;
		double C_AlGaSb_X = 0.0;
		double C_GaAsP_X  = 0.21;
		double C_AlSbAs_X = 0.0;
		double C_GaSbAs_X = 1.09;
		double C_InAsP_X  = 0.27;

		double C_AlGaAs_L = 0.055;
		double C_AlInAs_L = 0.0;
		double C_GaInAs_L = 0.72;
		double C_AlGaP_L  = 0.0;
		double C_AlInP_L  = 0.0;
		double C_GaInP_L  = 0.43;
		double C_AlGaSb_L = 0.55;
		double C_GaAsP_L  = 0.42;
		double C_AlSbAs_L = 0.0;
		double C_GaSbAs_L = 1.09;
		double C_InAsP_L  = 0.27;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		double C_GaInSb_G = 0.42;	//M Gudeny and J Piprek, Modelling Simul. Mater. Sci. Eng. 4 (1996) 349357. Printed in the UK
		double C_InSbAs_G = 0.58;	//Material parameters of quaternary IIIV semiconductors for multilayer mirrors at 1:55 m wavelength
		//     E_G  DEL_SO A    B
		//InSb 0.17 0.81   7.91 13.07

		double E_InSb_G  = 0.17;	//0.178 moze lepsze?
		double D_SO_InSb = 0.81;
		double A_InSb    = 7.91;
		double B_InSb    = 13.07;

		double E_InSb_X  = 4.078;	//Gonzalez-Cuevas et al., J. Appl. Phys. 102, 014504 2007.
		double E_InSb_L  = 1.858;	//Calculations of the temperature and alloy composition effects on the optical properties of AlxGa1-xAsySb1-y and GaxIn1-xAsySb1-y in the spectral range 0.56 eV
		double C_InSbAs_L = 0.55;
		double C_InSbAs_X = 0.60;
		double C_GaInSb_L = 0.38;
		double C_GaInSb_X = 0.24;
		//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		//*/

		//Binary alloy material parameters: direct and indirect energy bandgaps E_G,E_X, E_L [eV], splitting energies DEL_SO [eV] A, and strength parameters A and B
		/*
		AlAs 2.95 2.16 2.36 0.28 25.30		-0.8
		AlP  3.63 2.45 4.00 0.00 18.31		 0.59
		AlSb 2.30 1.61 2.21 0.72 59.68		-9.53
		GaAs 1.43 1.91 1.73 0.34 9.29(5.3*)  7.86(10.4*)
		GaP  2.74 2.27 3.00 0.13 22.25		 0.9
		GaSb 0.72 1.05 0.76 0.74 4.05		 12.66
		InAs 0.36 1.37 1.07 0.40 4.36(6.14*) 11.65(9.15*)
		InP  1.35 2.04 1.87 0.11 8.40		 6.60
		*/
		//* Values of A and B used in calculations for AlGaInAs system.

	//*
		double E_AlAs_G = 2.95;
		double E_AlP_G  = 3.63;
		double E_AlSb_G = 2.30;
		double E_GaAs_G = 1.43;
		double E_GaP_G  = 2.74;
		double E_GaSb_G = 0.72;
		double E_InAs_G = 0.36;
		double E_InP_G  = 1.35;

		double E_AlAs_X = 2.16;
		double E_AlP_X  = 2.45;
		double E_AlSb_X = 1.61;
		double E_GaAs_X = 1.91;
		double E_GaP_X  = 2.27;
		double E_GaSb_X = 1.05;
		double E_InAs_X = 1.37;
		double E_InP_X  = 2.04;

		double E_AlAs_L = 2.36;
		double E_AlP_L  = 4.00;
		double E_AlSb_L = 2.21;
		double E_GaAs_L = 1.73;
		double E_GaP_L  = 3.00;
		double E_GaSb_L = 0.76;
		double E_InAs_L = 1.07;
		double E_InP_L  = 1.87;

		double D_SO_AlAs = 0.28;
		double D_SO_AlP  = 0.00;
		double D_SO_AlSb = 0.72;
		double D_SO_GaAs = 0.34;
		double D_SO_GaP  = 0.13;
		double D_SO_GaSb = 0.74;
		double D_SO_InAs = 0.40;
		double D_SO_InP  = 0.11;

		double A_AlAs = 25.30;
		double A_AlP  = 18.31;
		double A_AlSb = 59.68;
		double A_GaAs = 9.29;
		double A_GaP  = 22.25;
		double A_GaSb = 4.05;
		double A_InAs = 4.36;
		double A_InP  = 8.40;

		double B_AlAs = -0.80;
		double B_AlP  = 0.59;
		double B_AlSb = -9.53;
		double B_GaAs = 7.86;
		double B_GaP  = 0.90;
		double B_GaSb = 12.66;
		double B_InAs = 11.65;
		double B_InP  = 6.60;

		//dla Ga_xIn_1-xAs_ySb_1-y -		A(x,y) = x*y*A_AC + x*(1-y)*A_AD +(1-x)*y*A_BC + (1-x)*(1-y)*A_BD
		double D_SO_xy	= mGa*mGa*D_SO_GaAs + mGa*(1-mAs)*D_SO_GaSb +(1-mGa)*mAs*D_SO_InAs + (1-mGa)*(1-mAs)*D_SO_InSb;
		double A_xy		= mGa*mGa*A_GaAs + mGa*(1-mAs)*A_GaSb +(1-mGa)*mAs*A_InAs + (1-mGa)*(1-mAs)*A_InSb;
		double B_xy		= mGa*mGa*B_GaAs + mGa*(1-mAs)*B_GaSb +(1-mGa)*mAs*B_InAs + (1-mGa)*(1-mAs)*B_InSb;

		double E_xy_G, E_xy_X, E_xy_L;

		if((mGa == 0.0) && (mAs == 0.0))		//dla InSb
		{
			E_xy_G	= E_InSb_G;
			E_xy_X	= E_InSb_X;
			E_xy_L	= E_InSb_L;
		}
		else if((mGa == 1.0) && (mAs == 1.0))	//dla GaAs
		{
			E_xy_G	= E_GaAs_G;
			E_xy_X	= E_GaAs_X;
			E_xy_L	= E_GaAs_L;
		}
		else if((mGa == 1.0) && (mAs == 0.0))	//dla GaSb
		{
			E_xy_G	= E_GaSb_G;
			E_xy_X	= E_GaSb_X;
			E_xy_L	= E_GaSb_L;
		}
		else if((mGa == 0.0) && (mAs == 1.0))	//dla InAs
		{
			E_xy_G	= E_InAs_G;
			E_xy_X	= E_InAs_X;
			E_xy_L	= E_InAs_L;
		}
		else if((mGa == 0.0) && (mAs != 0.0))	//dla InSbAs
		{
			E_xy_G	= mAs*E_InAs_G + (1-mAs)*E_InSb_G  - mAs*(1-mAs)*C_InSbAs_G;
			E_xy_X	= mAs*E_InAs_X + (1-mAs)*E_InSb_X  - mAs*(1-mAs)*C_InSbAs_X;
			E_xy_L	= mAs*E_InAs_L + (1-mAs)*E_InSb_L  - mAs*(1-mAs)*C_InSbAs_L;
		}
		else if((mGa != 0.0) && (mAs == 0.0))	//dla GaInSb
		{
			E_xy_G	= mGa*E_GaSb_G  + (1-mGa)*E_InSb_G  - mGa*(1-mGa)*C_GaInSb_G;
			E_xy_X	= mGa*E_GaSb_X  + (1-mGa)*E_InSb_X  - mGa*(1-mGa)*C_GaInSb_X;
			E_xy_L	= mGa*E_GaSb_L  + (1-mGa)*E_InSb_L  - mGa*(1-mGa)*C_GaInSb_L;
		}
		else if((mGa == 1.0) && (mAs != 0.0))	//dla GaSbAs
		{
			E_xy_G	= mAs*E_GaAs_G  + (1-mAs)*E_GaSb_G  - mAs*(1-mAs)*C_GaSbAs_G;
			E_xy_X	= mAs*E_GaAs_X  + (1-mAs)*E_GaSb_X  - mAs*(1-mAs)*C_GaSbAs_X;
			E_xy_L	= mAs*E_GaAs_L  + (1-mAs)*E_GaSb_L  - mAs*(1-mAs)*C_GaSbAs_L;
		}
		else if((mGa != 0.0) && (mAs == 1.0))	//dla GaInAs
		{
			E_xy_G	= mGa*E_GaAs_G  + (1-mGa)*E_InAs_G  - mGa*(1-mGa)*C_GaInAs_G;
			E_xy_X	= mGa*E_GaAs_X  + (1-mGa)*E_InAs_X  - mGa*(1-mGa)*C_GaInAs_X;
			E_xy_L	= mGa*E_GaAs_L  + (1-mGa)*E_InAs_L  - mGa*(1-mGa)*C_GaInAs_L;
		}
		else
		{
			//Ga_xIn_1-xAs_ySb_1-y -		Eg(x,y) = (x*(1-x)*((1-y)*E_ABD_x + y*E_ABC_x) + y*(1-y)*((1-x)*E_BCD_y + x*E_ACD_y)) / (x*(1-x)+y*(1-y))
			//								E_ABC_x = x*E_AB + (1-x)*E_AC - x*(1-x)*C_ABC
			//								itd.
			double E_GaInSb_G	= mGa*E_GaSb_G + (1-mGa)*E_InSb_G - mGa*(1-mGa)*C_GaInSb_G;
			double E_GaInAs_G	= mGa*E_GaAs_G + (1-mGa)*E_InAs_G - mGa*(1-mGa)*C_GaInAs_G;
			double E_InSbAs_G	= mAs*E_InAs_G + (1-mAs)*E_InSb_G - mAs*(1-mAs)*C_InSbAs_G;
			double E_GaSbAs_G	= mAs*E_GaAs_G + (1-mAs)*E_GaSb_G - mAs*(1-mAs)*C_GaSbAs_G;

			E_xy_G	= (mGa*(1-mGa)*((1-mAs)*E_GaInSb_G + mAs*E_GaInAs_G) + mAs*(1-mAs)*((1-mGa)*E_InSbAs_G + mGa*E_GaSbAs_G)) / (mGa*(1-mGa)+mAs*(1-mAs));


			double E_GaInSb_X	= mGa*E_GaSb_X + (1-mGa)*E_InSb_X - mGa*(1-mGa)*C_GaInSb_X;
			double E_GaInAs_X	= mGa*E_GaAs_X + (1-mGa)*E_InAs_X - mGa*(1-mGa)*C_GaInAs_X;
			double E_InSbAs_X	= mAs*E_InAs_X + (1-mAs)*E_InSb_X - mAs*(1-mAs)*C_InSbAs_X;
			double E_GaSbAs_X	= mAs*E_GaAs_X + (1-mAs)*E_GaSb_X - mAs*(1-mAs)*C_GaSbAs_X;

			E_xy_X	= (mGa*(1-mGa)*((1-mAs)*E_GaInSb_X + mAs*E_GaInAs_X) + mAs*(1-mAs)*((1-mGa)*E_InSbAs_X + mGa*E_GaSbAs_X)) / (mGa*(1-mGa)+mAs*(1-mAs));


			double E_GaInSb_L	= mGa*E_GaSb_L + (1-mGa)*E_InSb_L - mGa*(1-mGa)*C_GaInSb_L;
			double E_GaInAs_L	= mGa*E_GaAs_L + (1-mGa)*E_InAs_L - mGa*(1-mGa)*C_GaInAs_L;
			double E_InSbAs_L	= mAs*E_InAs_L + (1-mAs)*E_InSb_L - mAs*(1-mAs)*C_InSbAs_L;
			double E_GaSbAs_L	= mAs*E_GaAs_L + (1-mAs)*E_GaSb_L - mAs*(1-mAs)*C_GaSbAs_L;

			E_xy_L	= (mGa*(1-mGa)*((1-mAs)*E_GaInSb_L + mAs*E_GaInAs_L) + mAs*(1-mAs)*((1-mGa)*E_InSbAs_L + mGa*E_GaSbAs_L)) / (mGa*(1-mGa)+mAs*(1-mAs));
		}

		//wybieramy najmniejsze E_xy
		double E_xy		= E_xy_G;
		if(E_xy_X < E_xy) E_xy = E_xy_X;
		if(E_xy_L < E_xy) E_xy = E_xy_L;

		double Ksi0  = E_photon/E_xy;
		double KsiSO = E_photon/(E_xy+D_SO_xy);

		double H0 = 1.0;
		if((1-Ksi0)<0)	H0 = 0.0;
		else			H0 = sqrt(1 - Ksi0);

		double HSO = 1.0;
		if((1-KsiSO)<0) HSO = 0.0;
		else			HSO = sqrt(1 - KsiSO);

		double F_Ksi0 = (2 - sqrt(1 + Ksi0) - H0)/(Ksi0*Ksi0);
		double F_KsiSO = (2 - sqrt(1 + KsiSO) - HSO)/(KsiSO*KsiSO);

		double nR = sqrt(A_xy*(F_Ksi0 + 0.5*pow(E_xy/(E_xy+D_SO_xy), 1.5)*F_KsiSO) + B_xy);

		//zaleznosc temperaturowa na bazie pracy:
		//brak
		
		double dn_dT = 0.0;	//1/K

		return nR + dn_dT*(iT-300);
	}
	else
		return 0.;
		//*/
	//return 0.0;
}
//---------------------------------------------------------------------------
void GaInAsSb::setInt(std::string iInt)
{
	mInt = iInt;
}
//---------------------------------------------------------------------------
GaInAsSb::GaInAsSb(double iGa,double iAs)
{
	mGa = iGa;
	mIn = 1.-iGa;
	mAs = iAs;
	mSb = 1.-mAs;
	mInt = "nlin1";
}
//---------------------------------------------------------------------------
GaInAsSb::GaInAsSb()
{  
}
//---------------------------------------------------------------------------
GaInAsSb::~GaInAsSb()
{
}
