#include "pch.h"

#include "GaInP.h"
//------------------------------------------------------------------------------
double GaInP::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.a(iT) + mIn * mInP.a(iT) );
}
//------------------------------------------------------------------------------
double GaInP::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	if (iPoint == 'G') return ( mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 0.65 );
	else if (iPoint == 'X') return ( mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 0.20 );
	else if (iPoint == 'L') return ( mGa * mGaP.Eg(iT,iPoint) + mIn * mInP.Eg(iT,iPoint) - mGa * mIn * 1.03 );
	else return 0.;
}
//------------------------------------------------------------------------------
double GaInP::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( Eg(iT, 'G') );
}
//------------------------------------------------------------------------------
double GaInP::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.Dso() + mIn * mInP.Dso() );
}
//------------------------------------------------------------------------------
double GaInP::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	}
	return ( mGa * mGaP.me() + mIn * mInP.me() - mGa * mIn * 0.01854 );
}
//------------------------------------------------------------------------------
double GaInP::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	}
	return ( mGa * mGaP.mhh() + mIn * mInP.mhh() );
}
//------------------------------------------------------------------------------
double GaInP::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	}
	return ( mGa * mGaP.mlh() + mIn * mInP.mlh() );
}
//------------------------------------------------------------------------------
double GaInP::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.mso() + mIn * mInP.mso() );
}
//------------------------------------------------------------------------------
double GaInP::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double GaInP::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP; punktem odniesienia jest Ev=0eV dla InSb";
	}
	return ( mGa * mGaP.Ev() + mIn * mInP.Ev() );
}
//------------------------------------------------------------------------------
double GaInP::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double GaInP::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.ac() + mIn * mInP.ac() );
}
//------------------------------------------------------------------------------
double GaInP::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.av() + mIn * mInP.av() );
}
//------------------------------------------------------------------------------
double GaInP::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.b() + mIn * mInP.b() );
}
//------------------------------------------------------------------------------
double GaInP::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.d() + mIn * mInP.d() );
}
//------------------------------------------------------------------------------
double GaInP::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.c11() + mIn * mInP.c11() );
}
//------------------------------------------------------------------------------
double GaInP::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.c12() + mIn * mInP.c12() );
}
//------------------------------------------------------------------------------
double GaInP::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP";
	}
	return ( mGa * mGaP.c44() + mIn * mInP.c44() );
}
//------------------------------------------------------------------------------
double GaInP::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double GaInP::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double GaInP::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double GaInP::n(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	if (iDop == "Si")
	{
		double tn1 = Nc(iT) * exp ( -5.5e-3 / (cPhys::kB_eV*iT) ); // 5.5 meV -> Journal of Applied Physics 85, 7374 (1999)
		return ( (tn1/4.)*(sqrt(1.+8.*iN/tn1)-1.) ); // gD = 2
	}
	else if (iDop == "Zn")
	{
		double tp1 = Nv(iT) * exp ( -21e-3 / (cPhys::kB_eV*iT) ); // 21 meV -> R. Kdela, J. Novk, and M. Kuera "Zn-Doped InGaP Grown by the LP-MOCVD", Journal of Electronic Materials, 26, no.1, 1997. 
		return ( (tp1/8.)*(sqrt(1.+16.*iN/tp1)-1.) ); // gA = 4
	}
	else		
		return iN; // TODO
}
//------------------------------------------------------------------------------
double GaInP::mob(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if (iDop == "Si")
	{
		double tGaInP_miRT = 3190e-4/(1+pow(n(300., iN, iDop)/3e21,0.29)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "Se")
	{
		double tGaInP_miRT = 4740e-4/(1+pow(n(300., iN, iDop)/8e21,0.37)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "Sn")
	{
		double tGaInP_miRT = 71000e-4/(1+pow(n(300., iN, iDop)/1e21,0.95)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "C") // jaki to typ: n czy p?
	{
		double tGaInP_miRT = 580e-4/(1+pow(n(300., iN, iDop)/3e24,1.20)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,1.25) ); // (1.25 -> ref: praca magisterska Lukasza)		
	}
	else if (iDop == "Zn")
	{
		double tGaInP_miRT = 37e-4/(1+pow(n(300., iN, iDop)/3e25,0.63)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,0.5) ); // (0.5 -> ref: praca magisterska Lukasza)		  
	}
	else if (iDop == "Be")
	{
		double tGaInP_miRT = 270e-4/(1+pow(n(300., iN, iDop)/3e22,0.98)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,0.5) ); // (0.5 -> ref: praca magisterska Lukasza)		  
	}
	else if (iDop == "Cd")
	{
		double tGaInP_miRT = 84e-4/(1+pow(n(300., iN, iDop)/1e24,-0.18)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tGaInP_miRT * pow(300./iT,0.5) ); // (0.5 -> ref: praca magisterska Lukasza)		  
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double GaInP::ec(double iT, double iN, std::string iDop, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cPhys::e * mob(iT, iN, iDop) * n(iT, iN, iDop) );       
}
//------------------------------------------------------------------------------
double GaInP::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	} 
	return ( 1./(mGa/mGaP.k(iT) + mIn/mInP.k(iT) + mGa*mIn*0.72) );
}
//------------------------------------------------------------------------------
double GaInP::cp(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	}
	return ( mGa * mGaP.cp(iT) + mIn * mInP.cp(iT) );
}
//------------------------------------------------------------------------------
double GaInP::g(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla GaP i InP"; 
	}
	return ( mGa * mGaP.g(iT) + mIn * mInP.g(iT) );
}
//------------------------------------------------------------------------------
double GaInP::nR(double iLam, double iT, reference *ipRef) 
{
	if( (600.<=iLam) && (iLam<=690.))
	{
		double nR;
		if (mGa == 0.40) nR = 3.650;
		else if (mGa == 0.41) nR = 3.650; 
		else if (mGa == 0.42) nR = 3.651;
		else if (mGa == 0.43) nR = 3.652; 
		else if (mGa == 0.44) nR = 3.653;
		else if (mGa == 0.45) nR = 3.654; 
		else if (mGa == 0.46) nR = 3.656;
		else if (mGa == 0.47) nR = 3.659; 
		else if (mGa == 0.48) nR = 3.664;
		else if (mGa == 0.49) nR = 3.670; 
		else if (mGa == 0.50) nR = 3.679;                                
		else  
		{
			double A1 = 4.5714907083; // wspolczynniki dla GaP
			double B1 =-0.0028341929;
			double C1 = 1.3330522901e-6;
			double A2 = 5.8660721658; // wspolczynniki dla InP
			double B2 =-0.0063570685;
			double C2 = 4.2252822553e-6;
			double nR1 = A1+B1*iLam+C1*iLam*iLam; // zaleznosc dla GaP
			double nR2 = A2+B2*iLam+C2*iLam*iLam; // zaleznosc dla InP
			nR = mGa*nR1+(1.-mGa)*nR2;
		};  
		return ( nR + 2e-4*(iT-300.) );
	}
	else return 0.;
	/*if ( (iLam>=1100.) && (mIn<0.5) ) // T. Takagi, "Refractive Index of Ga(1-x)In(x)As Prepared by Vapor-Phase Epitaxy", Japanese Journal of Applied Physics, Vol. 17, No. 10, pp. 1813-1817, 1978.
	{
		double E0 = 3.65 - 2.15*mIn; 
		double Ed = 36.1 - 19.9*mIn;
		double Eg = 1.425- 1.337*mIn + 0.270*mIn*mIn;
		double E = cFunc::LamtoE(iLam);
		double eta = cMath::pi*Ed/(2.0*pow(E0,3.)*(E0*E0-Eg*Eg));    
		double nR300K = sqrt(1.+Ed/E0+Ed*E*E/pow(E0,3.)+(eta/cMath::pi)*pow(E,4.)*log((2.*E0*E0-Eg*Eg-E*E)/(Eg*Eg-E*E)));        
		return ( nR300K+2.5e-4*(iT-300.0) );    
	}
	else if ( ((iLam>=1200.)&&(2600.>=iLam)&&(mIn<=0.3)) ||	((iLam>=900.)&&(iLam<=1100.)&&(mIn<=0.1)) || ((iLam>=1600.)&&(iLam<=2600.)&&(mIn>=0.4)&&(mIn<=0.5)) || ((iLam>=2000.)&&(iLam<=2600.)&&(mIn>=0.6)&&(mIn<=0.7)) ) // http://www.batop.de/
	{
		double A = 8.950; // empirical coefficient 	 
		double B = 2.054; // empirical coefficient 	 
		double C = 0.6245; // empirical coefficient 	  
		double EgGaAs = 1.424; // fundamental band gap [eV] of GaAs at room temperature (300 K)                      
		double Eg = EgGaAs - mIn*1.501 + mIn*mIn*0.436;
		double nR300K = sqrt(A+B/(1.-pow(C*EgGaAs/(iLam*1.-3*Eg),2.)));
		return ( nR300K+2.5e-4*(iT-300.) );
	}
	else //if ((iLam>=1250.)&&(iLam<=1350.)&&(mIn==1.))
	{
		double nR300K = 3.8523305-iLam*0.0002365;
		return ( nR300K+2.5e-4*(iT-300.0) ); 
	}*/
}
//---------------------------------------------------------------------------
GaInP::GaInP(double iGa)
{
	mGa = iGa;
	mIn = 1.-iGa;
}
//---------------------------------------------------------------------------
GaInP::GaInP()
{  
}
//---------------------------------------------------------------------------
GaInP::~GaInP()
{
}