#include "pch.h"

#include "InGaAs.h"
//------------------------------------------------------------------------------
double InGaAs::cte(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="www.ioffe.ru/SVA/NSM/Semicond";
		ipRef->comment="wartosc dla In0.5Ga0.5As dla T = 300 K"; 
	} 
	return 5.09e-6;
}
//------------------------------------------------------------------------------
double InGaAs::a(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.a(iT) + mGa * mGaAs.a(iT) );
}
//------------------------------------------------------------------------------
double InGaAs::Eg(double iT, char iPoint, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	if (iPoint == 'G') return ( mIn * mInAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mIn * mGa * 0.477 );
	else if (iPoint == 'X') return ( mIn * mInAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mIn * mGa * 1.4 );
	else if (iPoint == 'L') return ( mIn * mInAs.Eg(iT,iPoint) + mGa * mGaAs.Eg(iT,iPoint) - mIn * mGa * 0.33 );
	else return 0.;
}
//------------------------------------------------------------------------------
double InGaAs::Eg(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( Eg(iT, 'G') );
}
//------------------------------------------------------------------------------
double InGaAs::Dso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.Dso() + mGa * mGaAs.Dso() - mIn * mGa * 0.15 );
}
//------------------------------------------------------------------------------
double InGaAs::me(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
	}
	return ( mIn * mInAs.me() + mGa * mGaAs.me() - mIn * mGa * 0.008 );
}
//------------------------------------------------------------------------------
double InGaAs::mhh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
	}
	return ( mIn * mInAs.mhh() + mGa * mGaAs.mhh() );
}
//------------------------------------------------------------------------------
double InGaAs::mlh(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
	}
	return ( mIn * mInAs.mlh() + mGa * mGaAs.mlh() );
}
//------------------------------------------------------------------------------
double InGaAs::mso(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.mso() + mGa * mGaAs.mso() );
}
//------------------------------------------------------------------------------
double InGaAs::mh(reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="mh = (mlh^3/2+mhh^3/2)^2/3"; 
	}
	return ( pow(pow(mlh(), 1.5) + pow(mhh(), 1.5), 2./3.) );
}
//------------------------------------------------------------------------------
double InGaAs::Ev(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs; punktem odniesienia jest Ev=0eV dla InSb";
	}
	return ( mIn * mInAs.Ev() + mGa * mGaAs.Ev() - mIn * mGa * (-0.38) );
}
//------------------------------------------------------------------------------
double InGaAs::Ec(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( Ev() + Eg(iT) );
}
//------------------------------------------------------------------------------
double InGaAs::ac(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.ac() + mGa * mGaAs.ac() - mIn * mGa * 2.61 );
}
//------------------------------------------------------------------------------
double InGaAs::av(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.av() + mGa * mGaAs.av() );
}
//------------------------------------------------------------------------------
double InGaAs::b(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.b() + mGa * mGaAs.b() );
}
//------------------------------------------------------------------------------
double InGaAs::d(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.d() + mGa * mGaAs.d() );
}
//------------------------------------------------------------------------------
double InGaAs::c11(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.c11() + mGa * mGaAs.c11() );
}
//------------------------------------------------------------------------------
double InGaAs::c12(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.c12() + mGa * mGaAs.c12() );
}
//------------------------------------------------------------------------------
double InGaAs::c44(reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="I. Vurgaftman, J. R. Meyer, and L. R. Ram-Mohan";
		ipRef->title="Band parameters for III-V compound semiconductors and their alloys";
		ipRef->journal="Journal of Applied Physics, vol. 89, no. 11, pp. 5815-5875, 2001";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
	}
	return ( mIn * mInAs.c44() + mGa * mGaAs.c44() );
}
//------------------------------------------------------------------------------
double InGaAs::Nc(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nc(iT, me()) );
}
//------------------------------------------------------------------------------
double InGaAs::Nv(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::Nv(iT, mh()) );
}
//------------------------------------------------------------------------------
double InGaAs::ni(double iT, reference *ipRef)
{ 
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cFunc::ni(iT, me(), mh(), Eg(iT)) );
}
//------------------------------------------------------------------------------
double InGaAs::n(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if (iDop == "Si")
		return ( 0.55*iN ); // (dopasowanie do danych eksp.: Lukasz)
	else if (iDop == "Zn")
		return ( 0.90*iN ); // (dopasowanie do danych eksp.: Lukasz)
	else		
		return iN; // TODO
}
//------------------------------------------------------------------------------
double InGaAs::mob(double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac"; 
	}
	if (iDop == "Si")
	{
		double tInP_miRT = 16700e-4/(1+pow(n(300., iN, iDop)/6e22,0.42)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tInP_miRT * pow(300./iT,1.59) ); // (1.59 -> ref: rozprawa Lukasza)		
	}
	else if (iDop == "Sn")
	{
		double tInP_miRT = 10600e-4/(1+pow(n(300., iN, iDop)/2e23,0.39)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tInP_miRT * pow(300./iT,1.59) ); // (1.59 -> ref: rozprawa Lukasza)		
	}
	else if (iDop == "Zn")
	{
		double tInP_miRT = 250e-4/(1+pow(n(300., iN, iDop)/6e23,0.34)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tInP_miRT * pow(300./iT,1.59) ); // (1.59 -> ref: rozprawa Lukasza)		
	}
	else if (iDop == "Be")
	{
		double tInP_miRT = 120e-4/(1+pow(n(300., iN, iDop)/2e25,0.39)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tInP_miRT * pow(300./iT,1.59) ); // (1.59 -> ref: rozprawa Lukasza)		
	}
	else if (iDop == "C")
	{
		double tInP_miRT = 570e-4/(1+pow(n(300., iN, iDop)/9e20,0.21)); // (m^2/(V*s)) (dopasowanie do danych eksp.: Lukasz)
		return ( tInP_miRT * pow(300./iT,1.59) ); // (1.59 -> ref: rozprawa Lukasza)		
	}
	else
		return 0.;
}
//------------------------------------------------------------------------------
double InGaAs::ec(double iT, double iN, std::string iDop, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="-";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="standardowy wzor"; 
	}
	return ( cPhys::e * mob(iT, iN, iDop) * n(iT, iN, iDop) );       
}
//------------------------------------------------------------------------------
double InGaAs::k(double iT, reference *ipRef)
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="S. Adachi";
		ipRef->title="Properties of Semiconductors Alloys";
		ipRef->journal="John Wiley and Sons, 2009";
		ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
	} 
	return ( 1./(mIn/mInAs.k(iT) + mGa/mGaAs.k(iT) + mIn*mGa*0.72) );
}
//------------------------------------------------------------------------------
double InGaAs::cp(double iT, reference *ipRef)
{
	double tcp = 0.;
	/*
	if(ipRef!=NULL)
	{
		if (mIn > 0.49 && mIn < 0.55)
		{
			ipRef->reset();
			ipRef->author="Maciej Kuc";
			ipRef->title="-";
			ipRef->journal="-";
			ipRef->comment="dla T: 300-1200 K"; 
			tcp = 286 + 0.0777*(iT - 300);
		}		
		else
		{
			ipRef->reset();
			ipRef->author="S. Adachi";
			ipRef->title="Properties of Semiconductors Alloys";
			ipRef->journal="John Wiley and Sons, 2009";
			ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
			tcp = mIn * mInAs.cp(iT) + mGa * mGaAs.cp(iT);
		}
	}
	*/
	if (mIn > 0.49 && mIn < 0.55)
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="Maciej Kuc";
			ipRef->title="-";
			ipRef->journal="-";
			ipRef->comment="dla T: 300-1200 K";
		}
		tcp = 286 + 0.0777*(iT - 300);
	}		
	else
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="S. Adachi";
			ipRef->title="Properties of Semiconductors Alloys";
			ipRef->journal="John Wiley and Sons, 2009";
			ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
		}
		tcp = mIn * mInAs.cp(iT) + mGa * mGaAs.cp(iT);
	}
	return ( tcp );
}
//------------------------------------------------------------------------------
double InGaAs::g(double iT, reference *ipRef)
{
	double tg = 0.;
	/*
	if(ipRef!=NULL)
	{
		if (mIn > 0.49 && mIn < 0.55)
		{
			ipRef->reset();
			ipRef->author="-";
			ipRef->title="-";
			ipRef->journal="www.ioffe.ru/SVA/NSM/Semicond/GaInAs";
			ipRef->comment="dla T: 200-470 K"; 
			tg = 5506 - 0.11325*(iT - 300);
		}
		else
		{
			ipRef->reset();
			ipRef->author="S. Adachi";
			ipRef->title="Properties of Semiconductors Alloys";
			ipRef->journal="John Wiley and Sons, 2009";
			ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs"; 
			tg = mIn * mInAs.g(iT) + mGa * mGaAs.g(iT);
		}
	}
	*/
	if (mIn > 0.49 && mIn < 0.55)
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="-";
			ipRef->title="-";
			ipRef->journal="www.ioffe.ru/SVA/NSM/Semicond/GaInAs";
			ipRef->comment="dla T: 200-470 K";
		}
		tg = 5506 - 0.11325*(iT - 300);
	}
	else
	{
		if(ipRef!=NULL)
		{
			ipRef->reset();
			ipRef->author="S. Adachi";
			ipRef->title="Properties of Semiconductors Alloys";
			ipRef->journal="John Wiley and Sons, 2009";
			ipRef->comment="wartosc interpolowana z danych dla InAs i GaAs";
		}
		tg = mIn * mInAs.g(iT) + mGa * mGaAs.g(iT);
	}
	return ( tg );
}
//------------------------------------------------------------------------------
double InGaAs::nR(double iLam, double iT, reference *ipRef) 
{
	if ( (iLam>=1100.) && (mIn<0.5) ) // T. Takagi, "Refractive Index of Ga(1-x)In(x)As Prepared by Vapor-Phase Epitaxy", Japanese Journal of Applied Physics, Vol. 17, No. 10, pp. 1813-1817, 1978.
	{
		double E0 = 3.65 - 2.15*mIn; 
		double Ed = 36.1 - 19.9*mIn;
		double Eg = 1.425- 1.337*mIn + 0.270*mIn*mIn;
		double E = cFunc::LamtoE(iLam);
		double eta = cMath::pi*Ed/(2.0*pow(E0,3.)*(E0*E0-Eg*Eg));    
		double nR300K = sqrt(1.+Ed/E0+Ed*E*E/pow(E0,3.)+(eta/cMath::pi)*pow(E,4.)*log((2.*E0*E0-Eg*Eg-E*E)/(Eg*Eg-E*E)));        
		return ( nR300K+2.5e-4*(iT-300.0) );    
	}
	else if ( ((iLam>=1200.)&&(2600.>=iLam)&&(mIn<=0.3)) ||	((iLam>=900.)&&(iLam<=1100.)&&(mIn<=0.1)) || ((iLam>=1600.)&&(iLam<=2600.)&&(mIn>=0.4)&&(mIn<=0.5)) || ((iLam>=2000.)&&(iLam<=2600.)&&(mIn>=0.6)&&(mIn<=0.7)) ) // http://www.batop.de/
	{
		double A = 8.950; // empirical coefficient 	 
		double B = 2.054; // empirical coefficient 	 
		double C = 0.6245; // empirical coefficient 	  
		double EgGaAs = 1.424; // fundamental band gap [eV] of GaAs at room temperature (300 K)                      
		double Eg = EgGaAs - mIn*1.501 + mIn*mIn*0.436;
		double nR300K = sqrt(A+B/(1.-pow(C*EgGaAs/(iLam*1.-3*Eg),2.)));
		return ( nR300K+2.5e-4*(iT-300.) );
	}
	else //if ((iLam>=1250.)&&(iLam<=1350.)&&(mIn==1.))
	{
		double nR300K = 3.8523305-iLam*0.0002365;
		return ( nR300K+2.5e-4*(iT-300.0) ); 
	}  
}
//------------------------------------------------------------------------------
double InGaAs::abs(double iLam, double iT, double iN, std::string iDop, reference *ipRef) 
{
	if(ipRef!=NULL)
	{
		ipRef->reset();
		ipRef->author="Lukasz Piskorski";
		ipRef->title="-";
		ipRef->journal="-";
		ipRef->comment="na podstawie danych eksperymentalnych z roznych prac";
	}
	char tType = '-';
	if ( (iDop == "C") || (iDop == "Zn") || (iDop == "Be") )
		tType = 'p';
	else if ( (iDop == "Si") || (iDop == "Te") || (iDop == "Se") || (iDop == "Sn") )
		tType = 'n';
	double tn = n(iT, iN, iDop) * 1e-6;
	double tAbs300 = 0.;
	if (tType == 'p')
	{
		if (iLam == 1300.)		tAbs300 = 60500. * pow(tn*1e-18 + 23.3, -0.54);
		else if (iLam == 1550.)	tAbs300 = 24000. * pow(tn*1e-18 + 9.7, -0.61);
		else if (iLam == 2330.)	tAbs300 = 63. * pow(tn*1e-18, -0.7);
		else if (iLam == 9000.)	tAbs300 = 250. * pow(tn*1e-18, -0.7);
	}
	else if (tType == 'n')
	{
		if (iLam == 1300.)		tAbs300 = 18600. * pow(tn*1e-18 - 3.1, -0.64);
		else if (iLam == 1550.)	tAbs300 = 7600. * pow(tn*1e-18, -2.0);
	}
	return ( tAbs300 );
}
//------------------------------------------------------------------------------
InGaAs::InGaAs(double iIn)
{
	mIn = iIn;
	mGa = 1. - iIn;
}
//------------------------------------------------------------------------------
InGaAs::InGaAs()
{  
}
//------------------------------------------------------------------------------
InGaAs::~InGaAs()
{
}
//------------------------------------------------------------------------------
