#ifndef ZPRSMESA_MATERIAL_HPP
#define ZPRSMESA_MATERIAL_HPP

#include <plask/material.hpp>

#include "MaterialDataBase/material.h"
#include "MaterialDataBase/GaN.h"
#include "MaterialDataBase/AlN.h"
#include "MaterialDataBase/InN.h"

struct RpsMaterial: public plask::Material
{
    std::string _name;

    plask::shared_ptr<material> original;

    RpsMaterial(const std::string& name, const plask::shared_ptr<material>& original): _name(name), original(original) {}

    RpsMaterial(const std::string& name, material*&& original): _name(name), original(original) {}

    std::string name() const override {
        return _name;
    }

    Material::Kind kind() const override { return plask::Material::GENERIC; }

    Material::ConductivityType condtype() const override { return plask::Material::CONDUCTIVITY_UNDETERMINED; }

    double lattC(double T, char x) const override {
        if (x == 'a') return original->a(T);
        else if (x == 'c') return original->c(T);
        throw plask::BadInput("lattC", "Wrong lattice parameter");
    }

//     double Eg(double T, double e, char point) const override {  }
//
//     double VB(double T, double e, char point, char hole) const override {  }
//
//     double Dso(double T, double e) const override {  }
//
//     double Mso(double T, double e) const override {  }
//
//     plask::Tensor2<double> Me(double T, double e, char point) const override {  }
//
//     plask::Tensor2<double> Mhh(double T, double e) const override {  }
//
//     plask::Tensor2<double> Mlh(double T, double e) const override {  }
//
//     plask::Tensor2<double> Mh(double T, double e) const override {  }
//
//     double ac(double T) const override {  }
//
//     double av(double T) const override {  }
//
//     double b(double T) const override {  }
//
//     double d(double T) const override {  }
//
//     double c11(double T) const override {  }
//
//     double c12(double T) const override {  }
//
//     double c44(double T) const override {  }
//
//     double eps(double T) const override {  }
//
//     double chi(double T, double e, char point) const override {  }
//
//     double dens(double T) const override {  }
//
//     double cp(double T) const override {  }

    plask::Tensor2<double> cond(double T) const override {
        return original->ec(T);
    }

    plask::Tensor2<double> thermk(double T, double h) const override {
        return original->k(T);
    }

    double nr(double wl, double T, double n = .0) const override {
        return original->nR(wl, T);
    }

    double absp(double wl, double T) const override {
        return original->abs(wl, T);
    }

    template <typename RPS>
    struct Factory: public plask::MaterialsDB::MaterialConstructor {
        Factory(const std::string& name): MaterialConstructor(name) {}
        plask::shared_ptr<plask::Material> operator()(const Material::Composition&, double) const override {
            static plask::shared_ptr<RPS> original = plask::make_shared<RPS>();
            return plask::make_shared<RpsMaterial>(materialName, original);
        }
        bool isAlloy() const override { return false; }
    };

    template<typename RPS>
    struct Register {
        Register(const std::string& name) {
            plask::MaterialsDB::getDefault().addSimple(plask::make_shared<Factory<RPS>>(name));
        }
    };
};


struct RpsSemiconductor: public RpsMaterial
{
    std::string dopant;
    double doping;

    RpsSemiconductor(const std::string& name, const std::string& dopant, double dc,
                     const plask::shared_ptr<material>& original): RpsMaterial(name, original),
                     dopant(dopant), doping(1e6*dc) {}

    RpsSemiconductor(const std::string& name, const std::string& dopant, double dc,
                     material*&& original): RpsMaterial(name, std::move(original)),
                     dopant(dopant), doping(1e6*dc) {}

    std::string name() const override {
        if (dopant != "") return _name + ":" + dopant;
        return _name;
    }

    bool isEqual(const plask::Material& oth) const override {
        auto other = static_cast<const RpsSemiconductor&>(oth);
        return original == other.original && dopant == other.dopant && doping == other.doping;
    }

    Material::Kind kind() const override { return plask::Material::SEMICONDUCTOR; }

//     double Nc(double T, double e, char point) const override {  }
//
//     double Nv(double T, double e, char point) const override {  }
//
//     double Ni(double T) const override {  }
//
//     double Nf(double T) const override {  }
//
//     double EactD(double T) const override {  }
//
//     double EactA(double T) const override {  }
//
//     plask::Tensor2<double> mob(double T) const override {  }

    plask::Tensor2<double> cond(double T) const override {
        double ec = original->ec(T, doping, dopant);
        if (ec) return ec;
        else return original->ec(T);
    }

    plask::Tensor2<double> thermk(double T, double h) const override {
        double k = original->k(T, doping, dopant);
        if (k) return k;
        else return original->k(T);
    }

    double nr(double wl, double T, double n = .0) const override {
        double nR = original->nR(wl, T, doping, dopant);
        if (nR) return nR;
        else return original->nR(wl, T);
    }

    double absp(double wl, double T) const override {
        double abs = original->abs(wl, T, doping, dopant);
        if (abs) return abs;
        else return original->abs(wl, T);
    }

};

struct RpsBinary: public RpsSemiconductor {

    template <typename RPS>
    struct Factory: public plask::MaterialsDB::MaterialConstructor
    {
        Factory(const std::string& name): MaterialConstructor(name) {}

        plask::shared_ptr<plask::Material> operator()(const Material::Composition&, double doping) const {
            auto parts = plask::splitString2(materialName, ':');
            static plask::shared_ptr<RPS> original = plask::make_shared<RPS>();
            return plask::make_shared<RpsSemiconductor>(parts.first, parts.second, doping, original);
        }

        bool isAlloy() const override { return false; }

    };

    template <typename RPS>
    struct Register {
        Register(const std::string& name) {
            plask::MaterialsDB::getDefault().addSimple(plask::make_shared<Factory<RPS>>(name));
        }
    };
};

template <typename RPS>
struct RpsNitride: public RpsSemiconductor {

    template <typename... Args>
    RpsNitride(Args... args): RpsSemiconductor(args...) {}

    plask::Tensor2<double> thermk(double T, double h) const override {
        return RPS(h).k(T, doping, dopant);
    }

    struct Factory: public plask::MaterialsDB::MaterialConstructor
    {
        Factory(const std::string& name): MaterialConstructor(name) {}

        plask::shared_ptr<plask::Material> operator()(const Material::Composition&, double doping) const override {
            auto parts = plask::splitString2(materialName, ':');
            static plask::shared_ptr<RPS> original = plask::make_shared<RPS>();
            return plask::make_shared<RpsNitride>(parts.first, parts.second, doping, original);
        }

        bool isAlloy() const override { return false; }

    };

    struct Register {
        Register(const std::string& name) {
            plask::MaterialsDB::getDefault().addSimple(plask::make_shared<Factory>(name));
        }
    };
};

struct RpsTertiary: public RpsSemiconductor {

    template <typename RPS>
    struct Factory: public plask::MaterialsDB::MaterialConstructor
    {
        const std::string par0;

        Factory(const std::string& name, const std::string& par0): MaterialConstructor(name), par0(par0) {}

        plask::shared_ptr<plask::Material> operator()(const Material::Composition& composition, double doping) const override {
            auto parts = plask::splitString2(materialName, ':');
            auto found = composition.find(par0);
            assert(found != composition.end());
            static plask::shared_ptr<RPS> original = plask::make_shared<RPS>(found->second);
            return plask::make_shared<RpsSemiconductor>(parts.first, parts.second, doping, original);
        }

        bool isAlloy() const override { return true; }

    };

    template <typename RPS>
    struct Register {
        Register(const std::string& name, const std::string& par0) {
            plask::MaterialsDB::getDefault().addAlloy(plask::make_shared<Factory<RPS>>(name, par0));
        }
    };
};

struct RpsQuarternary: public RpsSemiconductor {

    template <typename RPS>
    struct Factory: public plask::MaterialsDB::MaterialConstructor
    {
        const std::string par0, par1;

        Factory(const std::string& name, const std::string& par0, const std::string& par1): MaterialConstructor(name), par0(par0), par1(par1) {}

        plask::shared_ptr<plask::Material> operator()(const Material::Composition& composition, double doping) const override {
            auto parts = plask::splitString2(materialName, ':');
            auto found0 = composition.find(par0);
            auto found1 = composition.find(par1);
            assert(found0 != composition.end());
            assert(found1 != composition.end());
            static plask::shared_ptr<RPS> original = plask::make_shared<RPS>(found0->second, found1->second);
            return plask::make_shared<RpsSemiconductor>(parts.first, parts.second, doping, original);
        }

        bool isAlloy() const override { return true; }

    };

    template <typename RPS>
    struct Register {
        Register(const std::string& name, const std::string& par0, const std::string& par1) {
            plask::MaterialsDB::getDefault().addAlloy(plask::make_shared<Factory<RPS>>(name, par0, par1));
        }
    };
};


// Macros for easy registration


#define MSVC_BUG(MACRO, ARGS) MACRO ARGS  // name to remind that bug fix is due to MSVC :-)

#define _GET_MACRO_2(_1, _2, _3, _4, _5, _6, _7, _8, NAME,...) NAME
#define _GET_MACRO(...) MSVC_BUG(_GET_MACRO_2, (__VA_ARGS__))

#define _BIN0(m) _##m(#m)
#define _BIN1(m, d) _BIN0(m), _##m##_##d(#m ":" #d)
#define _BIN2(m, d, d1) _BIN1(m, d1), _##m##_##d(#m ":" #d)
#define _BIN3(m, d, d1, d2) _BIN2(m, d1, d2), _##m##_##d(#m ":" #d)
#define _BIN4(m, d, d1, d2, d3) _BIN3(m, d1, d2, d3), _##m##_##d(#m ":" #d)
#define _BIN5(m, d, d1, d2, d3, d4) _BIN4(m, d1, d2, d3, d4), _##m##_##d(#m ":" #d)
#define _BIN6(m, d, d1, d2, d3, d4, d5) _BIN5(m, d1, d2, d3, d4, d5), _##m##_##d(#m ":" #d)
#define _BIN7(m, d, d1, d2, d3, d4, d5, d6) _BIN6(m, d1, d2, d3, d4, d5, d6), _##m##_##d(#m ":" #d)
#define _BIN8(m, d, d1, d2, d3, d4, d5, d6, d7) _BIN7(m, d1, d2, d3, d4, d5, d6, d7), _##m##_##d(#m ":" #d)

#define _TER0(m, c) _##m(#m, #c)
#define _TER1(m, c, d) _TER0(m, c), _##m##_##d(#m ":" #d, #c)
#define _TER2(m, c, d, d1) _TER1(m, c, d1), _##m##_##d(#m ":" #d, #c)
#define _TER3(m, c, d, d1, d2) _TER2(m, c, d1, d2), _##m##_##d(#m ":" #d, #c)
#define _TER4(m, c, d, d1, d2, d3) _TER3(m, c, d1, d2, d3), _##m##_##d(#m ":" #d, #c)
#define _TER5(m, c, d, d1, d2, d3, d4) _TER4(m, c, d1, d2, d3, d4), _##m##_##d(#m ":" #d, #c)
#define _TER6(m, c, d, d1, d2, d3, d4, d5) _TER5(m, c, d1, d2, d3, d4, d5), _##m##_##d(#m ":" #d, #c)
#define _TER7(m, c, d, d1, d2, d3, d4, d5, d6) _TER6(m, c, d1, d2, d3, d4, d5, d6), _##m##_##d(#m ":" #d, #c)
#define _TER8(m, c, d, d1, d2, d3, d4, d5, d6, d7) _TER7(m, c, d1, d2, d3, d4, d5, d6, d7), _##m##_##d(#m ":" #d, #c)

#define _QTR0(m, c1, c2) _##m(#m, #c1, #c2)
#define _QTR1(m, c1, c2, d) _QTR0(m, c1, c2), _##m##_##d(#m ":" #d, #c1, #c2)
#define _QTR2(m, c1 ,c2, d, d1) _QTR1(m, c1 ,c2, d1), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR3(m, c1 ,c2, d, d1, d2) _QTR2(m, c1 ,c2, d1, d2), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR4(m, c1 ,c2, d, d1, d2, d3) _QTR3(m, c1 ,c2, d1, d2, d3), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR5(m, c1 ,c2, d, d1, d2, d3, d4) _QTR4(m, c1 ,c2, d1, d2, d3, d4), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR6(m, c1 ,c2, d, d1, d2, d3, d4, d5) _QTR5(m, c1 ,c2, d1, d2, d3, d4, d5), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR7(m, c1 ,c2, d, d1, d2, d3, d4, d5, d6) _QTR6(m, c1 ,c2, d1, d2, d3, d4, d5, d6), _##m##_##d(#m ":" #d, #c1 ,#c2)
#define _QTR8(m, c1 ,c2, d, d1, d2, d3, d4, d5, d6, d7) _QTR7(m, c1 ,c2, d1, d2, d3, d4, d5, d6, d7), _##m##_##d(#m ":" #d, #c1 ,#c2)

#define REGISTER_SIMPLE(m) RpsMaterial::Register<m> _BIN0(m)
#define REGISTER_BINARY_DOPED(m, ...) RpsMaterial::Register<m> MSVC_BUG(_GET_MACRO(__VA_ARGS__, _BIN8, _BIN7, _BIN6, _BIN5, _BIN4, _BIN3, _BIN2, _BIN1, _BIN0,), (m, __VA_ARGS__))
#define REGISTER_NITRIDE(m) RpsNitride<m>::Register _BIN0(m)
#define REGISTER_NITRIDE_DOPED(m, ...) RpsNitride<m>::Register _BIN0(m)
#define REGISTER_TERTIARY(m, c) RpsTertiary::Register<m> _TER0(m, c)
#define REGISTER_TERTIARY_DOPED(m, c, ...) RpsTertiary::Register<m> MSVC_BUG(_GET_MACRO(__VA_ARGS__, _TER8, _TER7, _TER6, _TER5, _TER4, _TER3, _TER2, _TER1, _TER0,), (m, c, __VA_ARGS__))
#define REGISTER_QUATERNARY(m, c1, c2) RpsQuarternary::Register<m> _QTR0(m, c1, c2)
#define REGISTER_QUATERNARY_DOPED(m, c1, c2, ...) RpsQuarternary::Register<m> MSVC_BUG(_GET_MACRO(__VA_ARGS__, _QTR8, _QTR7, _QTR6, _QTR5, _QTR4, _QTR3, _QTR2, _QTR1, _QTR0,), (m, c1, c2, __VA_ARGS__))


#endif //ZPRSMESA_MATERIAL_H
